<?php


namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\ApiFailedException;
use Cms\Bundle\Member\Exception\FieldException;
use Cms\Bundle\Member\Exception\FieldMissingException;
use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Model\Member\MemberUtils;
use Move\Http\RequestUtils;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Trait MemberActivateTrait
 * @package Cms\Bundle\Member
 */
trait MemberActivateTrait
{

    use MemberUtilsTrait;

    /**
     * @return \Cms\Bundle\Member\MemberLoginService
     */
    abstract public function getLoginService() : MemberLoginService;

    /**
     * Gestion de l'action d'activation du membre
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return \Cms\Client\Member\Model\MemberRestfulModel
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\FieldException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     */
    protected function handleActivated(
        ServerRequestInterface $request
    ) : MemberRestfulModel {
        // on récupère la clé et l'id qu'elle contient
        $activationKey = $this->getQueryActivateKey($request);
        if (!$activationKey) {
            throw new FieldMissingException('ak');
        }
        $memberId = MemberUtils::getMemberIdWithPasswordKey($activationKey);
        if ($memberId === null) {
            throw new FieldException('ak', MemberErrorCode::KEY_INVALID);
        }

        // récupération du member
        $memberModel = $this->getMemberModel(
            $this->getLoginService()->getMemberClient(),
            $memberId,
            true
        );

        // récupération du login pour ce scope
        $login = $memberModel->getLoginByScope($this->getLoginService()->getScopeId());

        // vérification de la clé envoyée
        if ($activationKey !== $login->reset_passwd_key) {
            throw new FieldException('ak', MemberErrorCode::KEY_INVALID);
        }

        // vérification si la clé a déjà été utilisée
        if ($login->reset_passwd_used) {
            throw new FieldException('ak', MemberErrorCode::KEY_ALREADY_USED);
        }

        try {
            // si tout est OK, on met à jour le statut et on modifie l'état de la key
            $this->getLoginService()->getMemberClient()->updateLoginMember(
                $memberId,
                $this->getLoginService()->getScopeId()
            );
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot save member info from API',
                $e,
                MemberErrorCode::SAVE_MEMBER_API_FAILED
            );
        }
        return $memberModel;
    }


    /**
     * extrait de la request la activate key, si aucune key dans la request on renvoi null
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return string|null
     */
    protected function getQueryActivateKey(ServerRequestInterface $request)
    {
        return RequestUtils::getQueryValue($request, 'ak');
    }

    /**
     * @param string $uri
     * @param string $activationKey
     * @return string
     */
    protected function decorateUri(string $uri, string $activationKey) : string
    {
        $uri .= '?ak=' . urlencode($activationKey);
        return $uri;
    }
}
