<?php

namespace Cms\Bundle\Newsletter;

use Cms\Client\Command\Member\Command\SetNewsletterCommand;
use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Client\Service\MemberService;
use Cms\Model\Member\MemberAttributeName;
use League\Route\Http\Exception\BadRequestException;
use Move\Command\Exception\BadParameterException;
use Move\Newsletter\Provider\MailjetConfirmAdapter;
use Move\Newsletter\Provider\ProviderInterface;
use Move\Newsletter\ProviderAwareInterface;
use Move\Newsletter\ProviderAwareTrait;
use Move\Specification\EmailSpecification;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Zend\Diactoros\Response\EmptyResponse;
use Zend\Diactoros\Response\JsonResponse;

/**
 * Class NewsletterController
 * @package AutoMag\Front\Controller
 */
class NewsletterController implements ProviderAwareInterface, LoggerAwareInterface
{

    use LoggerAwareTrait;
    use ProviderAwareTrait;

    /** @var \Cms\Client\Service\MemberService */
    private $memberService;

    /**
     * NewsletterController constructor.
     * @param \Cms\Client\Service\MemberService $memberService
     */
    public function __construct(MemberService $memberService)
    {
        $this->memberService = $memberService;
        $this->logger = new NullLogger();
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return \Cms\Client\Command\Member\Command\SetNewsletterCommand
     * @throws \League\Route\Http\Exception\BadRequestException
     */
    protected function generateCommand(ServerRequestInterface $request)
    {
        $dataset = $request->getParsedBody();
        if (empty($dataset['email'])
            || !(new EmailSpecification())->isSatisfiedBy($dataset['email'])
        ) {
            throw new BadRequestException('Votre e-mail est invalide');
        }

        // check de la constante de scope
        if (!\defined('CMS_SCOPE_ID')) {
            throw new \RuntimeException('Constante for scope must be defined');
        }

        // insert dans la bdd du cms
        return new SetNewsletterCommand(CMS_SCOPE_ID, $dataset['email'], true);
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return \Zend\Diactoros\Response\JsonResponse
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \League\Route\Http\Exception\BadRequestException
     */
    public function record(ServerRequestInterface $request)
    {
        $command = $this->generateCommand($request);
        $this->execCommand($command);

        return new JsonResponse([
            'msg' => 'Merci de confirmer votre inscription via le mail qui vient de vous être envoyé.',
        ], 202);
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return \Psr\Http\Message\ResponseInterface
     * @throws \Exception
     */
    public function confirm(ServerRequestInterface $request)
    {
        // récupération du mail à part de la clé de confirmation
        $queryParams = $request->getQueryParams();
        if ($this->newsletterProvider instanceof MailjetConfirmAdapter) {
            $email = $this->newsletterProvider->getEmailFromConfirmKey($queryParams['key']);
        }

        // récupération du member correspondant à l'email
        $member = $this->memberService->getMemberClient()->getItemByEmail($email);
        if (!$member instanceof MemberRestfulModel) {
            throw new BadParameterException('no member with this email address : ' . $email);
        }

        // envoi de la confirmation pour le provider
        if (!$this->newsletterProvider->update($email)) {
            throw new \Exception('error with mailjet API');
        }

        // check de la constante de scope
        if (!\defined('CMS_SCOPE_ID')) {
            throw new \RuntimeException('Constante for scope must be defined');
        }

        // mise à jour de l'attribut du member
        try {
            $member->setAttributes(CMS_SCOPE_ID, MemberAttributeName::VALID_DBOPTIN, true);
            $member = $this->memberService->getMemberClient()->saveItem($member);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage(), [
                'exception' => $e,
            ]);
            throw new \Exception('error while recording member info', null, $e);
        }

        return $this->handleConfirmResponse(new EmptyResponse(), $member);
    }

    /**
     * @param \Cms\Client\Command\Member\Command\SetNewsletterCommand $command
     * @return MemberRestfulModel
     * @throws \RuntimeException
     */
    protected function execCommand(SetNewsletterCommand $command) : MemberRestfulModel
    {
        try {
            $result = $this->memberService->getCommandBus()->handle($command);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage(), [
                'exception' => $e,
            ]);
            throw new \RuntimeException('error while recording newsletter info', null, $e);
        }

        // creation de la newsletter
        if ($this->newsletterProvider instanceof ProviderInterface) {
            NewsletterHelper::setNewsletterFromModel(
                $command->getScopeId(),
                $result,
                $this->getProvider(),
                $this->getCustomDataset($command)
            );
        }

        return $result;
    }

    /**
     * @param ResponseInterface $response
     * @param $member
     * @return ResponseInterface
     */
    protected function handleConfirmResponse(ResponseInterface $response, MemberRestfulModel $member)
    {
        return $response;
    }

    /**
     * @param \Cms\Client\Command\Member\Command\SetNewsletterCommand $command
     * @return array
     */
    protected function getCustomDataset(SetNewsletterCommand $command) : array
    {
        return [
            'signin_source' => $command->getRewardSource(),
            'signin_campaign' => $command->getRewardCampaign(),
        ];
    }
}
