<?php

namespace Cms\Cache;

use Move\Cache\CacheAwareTrait;
use Psr\Cache\CacheItemInterface;
use Psr\Cache\InvalidArgumentException;

/**
 * Class CacheHelperTrait
 * @package Cms\Cache
 */
trait CacheHelperTrait
{

    use CacheAwareTrait;

    /**
     * @param string $cacheKey
     * @param null|int|\DateInterval $cacheTTL
     * @param CacheItemInterface $cacheItem
     * @return bool
     * @throws \InvalidArgumentException
     */
    protected function hitFromCache($cacheKey, $cacheTTL, &$cacheItem) : bool
    {
        $cachePool = $this->getItemCachePool();
        try {
            if ($cachePool && ($cacheItem = $cachePool->getItem($cacheKey))) {
                if ($cacheItem->isHit()) {
                    return true;
                }
                $cacheItem->expiresAfter($cacheTTL ?: 300);
            }
        } catch (InvalidArgumentException $e) {
            throw new \InvalidArgumentException($e->getMessage(), null, $e);
        }
        return false;
    }

    /**
     * @param CacheItemInterface|null $cacheItem
     * @param mixed $data
     * @return bool
     */
    protected function setInCache($cacheItem, $data) : bool
    {
        // prise en charge du cache
        $cachePool = $this->getItemCachePool();
        if (!empty($cachePool) && $cacheItem instanceof CacheItemInterface) {
            $result = $cachePool->save($cacheItem->set($data));
            return $result;
        }
        return false;
    }

    /**
     * @param callable $action
     * @param string $cacheKey
     * @param int $cacheTTL
     * @return mixed
     * @throws \RuntimeException
     * @throws \Psr\Cache\InvalidArgumentException
     */
    protected function decorateWithCache(callable $action, string $cacheKey, int $cacheTTL)
    {
        /** @var \Psr\Cache\CacheItemInterface $cacheItem */
        if (!$this->hitFromCache($cacheKey, $cacheTTL, $cacheItem)) {
            try {
                $result = $action();
            } catch (\Exception $e) {
                throw new \RuntimeException('cannot execute action for caching', null, $e);
            }
            $this->setInCache($cacheItem, $result);
            return $result;
        }
        return $cacheItem->get();
    }
}
