<?php

namespace Cms\Search\Contract;

use Cms\LinkResource\LinkResourceType;
use Cms\Model\Contract\ContractModel;
use Cms\Model\Contract\Product\ProductConfigI18nModel;
use Cms\Model\Contract\Product\ProductConfigModel;
use Cms\Model\Contract\Product\ProductConfigResourceModel;
use Cms\Model\Contract\Product\ProductModel;
use Cms\Model\Contract\Product\ProductProfile;
use Cms\Model\Contract\Product\ProductResourceModel;
use Cms\Search\Mapping\AddressDocumentTrait;
use Cms\Search\Mapping\AddressNestedMapping;
use Elastica\Document;
use Move\I18n\I18nLanguageCode;
use Move\Search\Analyzer\FrenchSearchAnalyzer;
use Move\Search\Analyzer\FrenchTextAnalyzer;
use Move\Search\ElasticMappingBuilder;
use Move\Search\Mapping\ElasticMappingProperty;
use Move\Search\Mapping\KeywordMapping;
use Move\Search\Mapping\ObjectMapping;
use Move\Search\Mapping\TextMapping;

/**
 * Class ContractIndexFactory
 * @package Cms\Search\Contract
 */
class ContractIndexFactory
{

    use AddressDocumentTrait;

    /** Nom du type pour les contract dans l'index */
    const MAPPING_NAME = 'cms_contract';

    /**
     * @return \Move\Search\ElasticMappingBuilder
     */
    public static function createMappingBuilder() : ElasticMappingBuilder
    {
        $builder = new ElasticMappingBuilder(self::MAPPING_NAME, [
            new ElasticMappingProperty('id', 'long'),
            new ElasticMappingProperty('scope_id', 'long'),

            (new KeywordMapping('state')),

            new ElasticMappingProperty('begin_at', 'date', false),
            new ElasticMappingProperty('end_at', 'date', false),

            (new ObjectMapping('advertiser', [
                new ElasticMappingProperty('id', 'long'),
                (new KeywordMapping('advertiser_type')),
                (new TextMapping('advertiser_label', false))
                    ->withAnalyzer(
                        new FrenchTextAnalyzer(),
                        new FrenchSearchAnalyzer()
                    )
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),
                new ElasticMappingProperty('advertiser_has_address', 'boolean'),
                new AddressNestedMapping(false),
            ]))->isNested(true),

            (new ObjectMapping('product_configs', [
                new ElasticMappingProperty('id', 'long'),
                new ElasticMappingProperty('product_id', 'long'),

                (new KeywordMapping('resource_provider')),
                (new KeywordMapping('resource_type')),
                (new KeywordMapping('resource_ref')),

                (new KeywordMapping('state')),
                (new KeywordMapping('profil')),

                new ElasticMappingProperty('begin_at', 'date', false),
                new ElasticMappingProperty('end_at', 'date', false),

                (new ObjectMapping('config_resources', [
                    new ElasticMappingProperty('id', 'long'),
                    new ElasticMappingProperty('config_id', 'long'),

                    (new KeywordMapping('language')),

                    (new KeywordMapping('resource_provider')),
                    (new KeywordMapping('resource_type')),
                    (new KeywordMapping('resource_ref')),
                ], false))->isNested(true),
            ], false))->isNested(true),

            (new ObjectMapping('product_resources', [
                new ElasticMappingProperty('id', 'long'),
                new ElasticMappingProperty('product_id', 'long'),

                (new KeywordMapping('resource_provider')),
                (new KeywordMapping('resource_type')),
                (new KeywordMapping('resource_ref')),
            ], false))->isNested(true),

            (new ObjectMapping('products', [
                new ElasticMappingProperty('id', 'long'),
                (new KeywordMapping('profil')),
            ], false))->isNested(true),

            new ElasticMappingProperty('created_at', 'date'),
            new ElasticMappingProperty('modified_at', 'date'),
        ]);
        return $builder;
    }

    /**
     * @param ContractModel $contractModel
     * @param \Cms\Model\Contract\Product\ProductModel[] $products
     * @param string $advertiserLabel
     * @param \Cms\Model\AddressBook\Address\AddressModel[] $addresses
     * @param \Cms\Model\Contract\Product\ProductResourceModel[] $resources
     * @param \Cms\Model\Contract\Product\ProductConfigModel[] $configs
     * @param \Cms\Model\Contract\Product\ProductConfigResourceModel[] $configResources
     * @param \Cms\Model\Contract\Product\ProductConfigI18nModel[] $configI18n
     * @return \Elastica\Document
     */
    public static function createDocument(
        ContractModel $contractModel,
        array $products,
        string $advertiserLabel,
        array $addresses = [],
        array $resources = [],
        array $configs = [],
        array $configResources = [],
        array $configI18n = []
    ) : Document {

        // ajouts address
        foreach ($addresses as $type => &$addressModel) {
            if (!\is_string($type)) {
                continue;
            }
            $addressModel = self::buildAddressMapping($type, $addressModel);
        }
        unset($addressModel);

        $documentData = [
            'id' => $contractModel->id,
            'scope_id' => $contractModel->scope_id,
            'state' => (string)$contractModel->state,
            'begin_at' => $contractModel->begin_at ? $contractModel->begin_at->format(DATE_W3C) : null,
            'end_at' => $contractModel->end_at ? $contractModel->end_at->format(DATE_W3C) : null,
            'created_at' => $contractModel->created_at->format(DATE_W3C),
            'modified_at' => $contractModel->modified_at->format(DATE_W3C),

            // ajout de l'annonceur
            'advertiser' => $contractModel->link_provider ? [
                'id' => (int)$contractModel->link_ref,
                'advertiser_type' => (string)$contractModel->link_type,
                'advertiser_label' => $advertiserLabel,
                // addresses de l'annonceur
                'advertiser_has_address' => !empty($addresses),
                'addresses_nested' => $addresses,
            ] : [],

            'product_configs' => [],

            'product_resources' => [],

            'products' => [],
        ];

        // ajout des produits
        $productProfil = [];
        foreach ($products as $product) {
            if (!$product instanceof ProductModel) {
                continue;
            }
            $productProfil[$product->id] = (string)$product->profil;
            $documentData['products'][] = [
                'id' => $product->id,
                'profil' => (string)$product->profil,
            ];
        }

        // ajout des config de produits
        foreach ($configs as $config) {
            if (!$config instanceof ProductConfigModel
                || !isset($productProfil[$config->product_id])
            ) {
                continue;
            }

            // init
            $resourcesInConfig = [];
            $productProfilType = $productProfil[$config->product_id];

            // patch pour logopartner il faut que la resource du partner
            if ($productProfilType === ProductProfile::LOGO_PARTENAIRE
                && (string)$contractModel->link_type === LinkResourceType::CAMPSITE_GROUP
            ) {
                $resourcesInConfig[] = [
                    'config_id' => $config->id,
                    'resource_type' => (string)$contractModel->link_type,
                    'resource_ref' => (string)$contractModel->link_ref,
                    'resource_provider' => (string)$contractModel->link_provider,
                ];
            } elseif (!empty($configResources[$config->id])) {
                // récuperation de la liste des config_resources
                foreach ($configResources[$config->id] as $configResource) {
                    if (!$configResource instanceof ProductConfigResourceModel) {
                        continue;
                    }
                    $resourcesInConfig[] = [
                        'id' => $configResource->id,
                        'config_id' => $config->id,
                        'resource_type' => (string)$configResource->link_type,
                        'resource_ref' => (string)$configResource->link_ref,
                        'resource_provider' => (string)$configResource->link_provider,
                    ];
                }
            }

            // récuperation de la liste des langue dispo
            $languagesInConfig = [];
            if (!empty($config->config)) {
                $languagesInConfig[] = I18nLanguageCode::FR;
            }
            if (!empty($configI18n[$config->id])) {
                foreach ($configI18n[$config->id] as $configLang) {
                    if (!$configLang instanceof ProductConfigI18nModel) {
                        continue;
                    }
                    $languagesInConfig[] = (string)$configLang->language;
                }
            }

            $documentData['product_configs'][] = [
                'id' => $config->id,
                'product_id' => $config->product_id,
                'resource_type' => (string)$config->link_type,
                'resource_ref' => (string)$config->link_ref,
                'resource_provider' => (string)$config->link_provider,
                'state' => (string)$config->state,
                'begin_at' => $config->begin_at ? $config->begin_at->format(DATE_W3C) : null,
                'end_at' => $config->end_at ? $config->end_at->format(DATE_W3C) : null,
                'profil' => $productProfilType,
                'config_resources' => $resourcesInConfig,
                'language' => $languagesInConfig,
            ];
        }

        // ajout des resources de produits
        foreach ($resources as $resource) {
            if (!$resource instanceof ProductResourceModel) {
                continue;
            }
            $documentData['product_resources'][] = [
                'id' => $resource->id,
                'product_id' => $resource->product_id,
                'resource_type' => (string)$resource->link_type,
                'resource_ref' => (string)$resource->link_ref,
                'resource_provider' => (string)$resource->link_provider,
            ];
        }


        // envoi des données
        $document = new Document($contractModel->id, $documentData, self::MAPPING_NAME);
        return $document;
    }
}
