<?php

namespace Move\Http;

use Dflydev\FigCookies\FigResponseCookies;
use Dflydev\FigCookies\SetCookie;
use Move\Image\ImageUtils;
use Psr\Http\Message\ResponseInterface;
use Zend\Diactoros\Stream;

/**
 * Class ResponseUtils
 * @package Move\Http
 */
class ResponseUtils
{

    /**
     * @param ResponseInterface $response
     * @param string $message
     * @param string $pngFilePath
     * @param array $vars
     * @return ResponseInterface
     */
    public static function withPlaceholderImageResponse(
        ResponseInterface $response,
        $message,
        $pngFilePath,
        $vars = []
    ) {
        // recuperation de l'image placeholder
        if (!isset($vars['height'])) {
            $vars['height'] = null;
        }
        if (!isset($vars['width'])) {
            $vars['width'] = null;
        }
        $img = ImageUtils::createPlaceholder($pngFilePath, $vars['width'], $vars['height']);

        // affichage de l'image
        $stream = new Stream('php://memory', 'w+');
        $stream->write(ImageUtils::imageToString($img, 'image/png'));
        $response = $response
            ->withStatus(200)
            ->withBody($stream)
            ->withHeader('X-Message', $message)
            ->withHeader('Content-Length', (string)$stream->getSize())
            ->withHeader('Content-Type', 'image/png');

        // ajoute le cors
        $response = self::withCORSHeaders($response, '*');

        return $response;
    }

    /**
     * Ajoute les headers pour no cache response
     * @param ResponseInterface $response
     * @return ResponseInterface
     */
    public static function withNoCache(ResponseInterface $response)
    {
        $headers = self::getNoCacheHeaders();
        return self::withArrayHeaders($response, $headers);
    }

    /**
     * Renvoi les headers pour no cache response
     * @return array
     */
    public static function getNoCacheHeaders()
    {
        return [
            'Expires' => 'Sat, 26 Jul 1997 05:00:00 GMT',
            'Cache-Control' => 'no-store, no-cache, must-revalidate',
        ];
    }

    /**
     * Ajoute un tableau de header
     * @param ResponseInterface $response
     * @param array $assocHeaders
     * @return ResponseInterface
     */
    protected static function withArrayHeaders(ResponseInterface $response, array $assocHeaders)
    {
        foreach ($assocHeaders as $name => $value) {
            $response = $response->withHeader($name, $value);
        }
        return $response;
    }

    /**
     * @param ResponseInterface $response
     * @param string $origin
     * @param array $allowHeaders
     * @param array $allowMethods
     * @param bool|false $allowCredential
     * @param int|null $maxAge
     * @return \Psr\Http\Message\MessageInterface|ResponseInterface
     */
    public static function withCORSHeaders(
        ResponseInterface $response,
        $origin,
        array $allowHeaders = [],
        array $allowMethods = [],
        $allowCredential = false,
        $maxAge = null
    ) {

        $headers = self::getCORSHeaders($origin, $allowHeaders, $allowMethods, $allowCredential, $maxAge);
        return self::withArrayHeaders($response, $headers);
    }

    /**
     * @param string $origin
     * @param array $allowHeaders
     * @param array $allowMethods
     * @param bool|false $allowCredential
     * @param int|null $maxAge
     * @return array
     */
    public static function getCORSHeaders(
        $origin,
        array $allowHeaders = [],
        array $allowMethods = [],
        $allowCredential = false,
        $maxAge = null
    ) {
        $headers = [];
        if (!empty($allowHeaders)) {
            $headers['Access-Control-Allow-Headers'] = $allowHeaders;
        }

        if (!empty($allowMethods)) {
            $headers['Access-Control-Allow-Methods'] = $allowMethods;
        }

        if (!empty($maxAge) && is_numeric($maxAge)) {
            $headers['Access-Control-Max-Age'] = (string)$maxAge;
        }
        if ($allowCredential === true) {
            $headers['Access-Control-Allow-Credentials'] = 'true';
        }

        $headers['Access-Control-Allow-Origin'] = $origin;

        return $headers;
    }

    /**
     * Ajoute les headers de last-modified + expires
     * @param ResponseInterface $response
     * @param int $lastModifiedTime
     * @param int|null $timeToLive
     * @return ResponseInterface
     */
    public static function withLastModified(ResponseInterface $response, $lastModifiedTime, $timeToLive = null)
    {
        $headers = self::getLastModifiedHeaders($lastModifiedTime, $timeToLive);
        return self::withArrayHeaders($response, $headers);
    }

    /**
     * Renvoi les headers de last-modified + expires
     * @param int $lastModifiedTime
     * @param int|null $timeToLive
     * @return array
     */
    public static function getLastModifiedHeaders($lastModifiedTime, $timeToLive = null)
    {
        $headers['Last-Modified'] = gmdate('D, d M Y H:i:s T', $lastModifiedTime);
        if (is_numeric($timeToLive) && $timeToLive > 0) {
            $headers['Cache-Control'] = ['public', 'max-age=' . $timeToLive];
            $headers['Expires'] = gmdate('D, d M Y H:i:s T', time() + $timeToLive);
        }
        return $headers;
    }

    /**
     * @param ResponseInterface $response
     * @param string $name
     * @return ResponseInterface
     */
    public static function RemoveCookie(ResponseInterface $response, $name)
    {
        // supppression du cookie
        $cookie = SetCookie::create($name)
            ->withValue(null)
            ->withPath('/')
            ->withExpires(time() - 3600 * 24);

        return FigResponseCookies::set($response, $cookie);
    }
}
