<?php


namespace Move\Http\Strategy;

use League\Route\Http\Exception\HttpExceptionInterface;
use League\Route\Http\Exception\NotAcceptableException;
use League\Route\Route;
use League\Route\Strategy\AbstractStrategy;
use League\Route\Strategy\StrategyInterface;
use Move\Http\Strategy\Handler\HandlerServiceInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LogLevel;
use Psr\Log\NullLogger;

/**
 * Class FormatStrategy
 * @package Move\Http\Strategy
 */
class FormatStrategy extends AbstractStrategy implements
    StrategyInterface,
    LoggerAwareInterface
{

    use LoggerAwareTrait;


    /** si aucun handler n'est trouvé une exception est lancée */
    public const REQUIRED_POLICY = 'require';

    /** Si aucun handler n'est trouvé on execute quand meme */
    public const BYPASS_POLICY = 'bypass';


    /**
     * @var HandlerServiceInterface|null
     */
    private $handlerFallback;

    /**
     * @var HandlerServiceInterface[]
     */
    private $handlerCollection;

    /**
     * @var string
     */
    private $handlerPolicy;

    /**
     * FormatStrategy constructor.
     * @param HandlerServiceInterface[] $handlerCollection
     * @param HandlerServiceInterface|null $handlerFallback
     * @throws \InvalidArgumentException
     */
    public function __construct(
        $handlerCollection,
        HandlerServiceInterface $handlerFallback = null
    ) {
        $this->handlerCollection = $handlerCollection;
        $this->setHandlerFallback($handlerFallback ?: end($handlerCollection));
        $this->handlerPolicy = self::BYPASS_POLICY;
        $this->logger = new NullLogger();
    }

    /**
     * @param string $policy
     */
    public function handlerRequirePolicy($policy = self::REQUIRED_POLICY)
    {
        $this->handlerPolicy = $policy;
    }

    /**
     * @param callable $controller
     * @param array $vars
     * @param Route|null $route
     * @return ResponseInterface
     * @throws \Exception
     */
    public function dispatch(callable $controller, array $vars, Route $route = null)
    {
        // test des strategy de reponse
        foreach ($this->handlerCollection as $handler) {
            if (true === $handler->isSatisfiedBy($this->getRequest())) {
                $applyHandler = $handler;
                break;
            }
        }

        // check de la politique de handler
        if ($this->handlerPolicy === self::REQUIRED_POLICY) {
            if (!isset($applyHandler)) {
                throw new NotAcceptableException();
            }
        } elseif (empty($applyHandler)) {
            $applyHandler = $this->handlerFallback;
        }

        // récuperation de l'objet reponse
        $httpResponse = $this->getResponse();
        $httpRequest = $this->getRequest();
        try {
            // call du controller, les données sont renvoyé sous forme de donnée brut
            // ou sous forme de ResponseFactory ou ResponseInterface
            $response = $controller($httpRequest, $httpResponse, $vars);

            // traitement du retour
            if ($applyHandler instanceof HandlerServiceInterface) {
                $response = $this->convertDataset(
                    $response,
                    $applyHandler,
                    $httpResponse
                );
            }
        } catch (\Exception $e) {
            // log des erreurs serveurs uniquement
            if (!$e instanceof HttpExceptionInterface || $e->getStatusCode() >= 500) {
                $this->logException($e, LogLevel::ERROR, $httpRequest);
            } else {
                // les erreur 40x doivent être géré par le client
                $this->logException($e, LogLevel::DEBUG, $httpRequest);
            }
            // traitement de l'erreur
            if ($applyHandler instanceof HandlerServiceInterface) {
                $response = $this->convertDataset(
                    $applyHandler->handleException($e),
                    $applyHandler,
                    $httpResponse
                );
            } else {
                throw $e;
            }
        }

        // renvoi de la reponse
        return $this->determineResponse($response);
    }

    /**
     * @param mixed $dataset
     * @param \Move\Http\Strategy\Handler\HandlerServiceInterface $applyHandler
     * @param \Psr\Http\Message\ResponseInterface $httpResponse
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \UnexpectedValueException
     */
    private function convertDataset(
        $dataset,
        HandlerServiceInterface $applyHandler,
        ResponseInterface $httpResponse
    ) {
        if (!$dataset instanceof ResponseInterface) {
            $httpResponse = $applyHandler->determineResponse($dataset, $httpResponse);
            if (!$httpResponse instanceof ResponseInterface) {
                throw new \UnexpectedValueException('dataHandler must return ResponseInterface object');
            }
            return $httpResponse;
        }
        return $dataset;
    }

    /**
     * @param \Exception $e
     * @param mixed $level
     * @param \Psr\Http\Message\ServerRequestInterface $request
     */
    private function logException(\Exception $e, $level, ServerRequestInterface $request)
    {
        $type = 'FormatStrategyError';
        $message = vsprintf($type . ' "%s" in %s:%s', [
            $e->getMessage(),
            $e->getFile(),
            $e->getLine(),
        ]);

        $previous = $e;
        do {
            $messages[] = sprintf(
                "%s:%d %s (%d) [%s]\n",
                $previous->getFile(),
                $previous->getLine(),
                $previous->getMessage(),
                $previous->getCode(),
                \get_class($previous)
            );
        } while ($previous = $previous->getPrevious());

        $this->logger->log($level, $message, [
            'request_uri' => (string)$request->getUri(),
            'request_meth' => $request->getMethod(),
            'err_message' => $e->getMessage(),
            'err_class' => \get_class($e),
            'err_type' => $type,
            'err_message_stack' => $messages,
            'exception' => $e,
        ]);
    }

    /**
     * @param HandlerServiceInterface|null $handlerFallback
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function setHandlerFallback($handlerFallback)
    {
        $this->handlerFallback = $this->validateHandler($handlerFallback);
        return $this;
    }

    /**
     * @param HandlerServiceInterface $handler
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function addHandler($handler)
    {
        $this->handlerCollection[] = $this->validateHandler($handler);
        return $this;
    }

    /**
     * @param mixed $handler
     * @return HandlerServiceInterface
     * @throws \InvalidArgumentException
     */
    final protected function validateHandler($handler)
    {
        if (null !== $handler && !$handler instanceof HandlerServiceInterface) {
            throw new \InvalidArgumentException('handler must be null or instance of HandlerServiceInterface');
        }
        return $handler;
    }
}
