<?php

namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\ApiFailedException;
use Cms\Bundle\Member\Exception\FieldException;
use Cms\Bundle\Member\Exception\MemberNotFoundException;
use Cms\Bundle\Member\Exception\RuntimeException;
use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Model\Member\MemberLoginModel;
use Move\Http\ResponseUtils;
use Move\Http\Strategy\DataContainer\EngineDataContainer;
use Move\Http\Strategy\ResponseFactory;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class AbstractCreateMemberController
 * @package Cms\Bundle\Member
 */
abstract class AbstractCreateMemberController extends AbstractSaveMemberController
{

    use MemberActivateTrait;

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Move\Http\Strategy\ResponseFactory|mixed
     */
    public function index(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        $data = $this->getEngineContainer($request->getQueryParams());
        $factory = new ResponseFactory($data, 200, $response->getHeaders());
        $factory->mergeHeaders(ResponseUtils::getLastModifiedHeaders(time(), 120));
        return $factory;
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Psr\Http\Message\ResponseInterface|mixed
     * @throws \InvalidArgumentException
     * @throws \Cms\Bundle\Member\Exception\FieldException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\PasswordFailedException
     * @throws \Cms\Bundle\Member\Exception\RuntimeException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     */
    public function record(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        $this->validateRequest($request);

        // filtrage des data en params
        $dataset = $request->getParsedBody();
        $params = $this->filterRequestForParams($dataset);

        // on check la validité du mot de passe
        $this->validatePassword($dataset);

        // vérification d'unicité
        try {
            $memberModel = $this->getMemberModel(
                $this->getLoginService()->getMemberClient(),
                $dataset['email'],
                true
            );
        } catch (MemberNotFoundException $e) {
            $memberModel = null;
        }

        // si pas de membre et pas de compte, on crée
        if (!$memberModel) {
            $memberModel = (new MemberRestfulModel())->populate($dataset);
            $memberModel->password_scope_id = $this->getLoginService()->getScopeId();

            // ajout des attributs
            $scopeId = $this->getLoginService()->getScopeId();
            foreach ($params as $attr => $val) {
                $memberModel->setAttributes($scopeId, $attr, $val, true);
            }

            // sauvegarde du membre
            $memberModel = $this->handleMemberRecord($memberModel);
        } else {
            // récupération des scopes
            $scopeId = $this->getLoginService()->getScopeId();
            $login = $memberModel->getLoginByScope($scopeId);
            // check si le membre n'a pas déjà de compte
            if ($login === null) {
                // ajout des attributs
                foreach ($params as $attr => $val) {
                    $memberModel->setAttributes($scopeId, $attr, $val, true);
                }

                // sauvegarde du membre
                $memberModel = $this->handleMemberRecord($memberModel);

                // sauvegarde du LoginMember
                try {
                    $memberModel = $this->getLoginService()->getMemberClient()->createLoginMember(
                        $memberModel->id,
                        $scopeId,
                        $dataset['password']
                    );
                } catch (\Exception $e) {
                    throw new ApiFailedException(
                        'cannot save member info for email ' . $dataset['email'],
                        $e,
                        MemberErrorCode::SAVE_MEMBER_API_FAILED
                    );
                }
            } else {
                // si déjà un compte, envoi du message d'erreur
                throw new FieldException(
                    'email',
                    MemberErrorCode::EMAIL_ALREADY_EXIST
                );
            }
        }

        // envoi du mail de compte créé avec demande de confirmation
        // récupération du login pour ce scope
        $login = $memberModel->getLoginByScope($this->getLoginService()->getScopeId());
        if ($login === null || !$login->reset_passwd_key) {
            throw new RuntimeException('activation key missing');
        }

        // envoi du mail
        $email = $this->resolvEmail($memberModel->email);
        $this->sendActivationMail($email, $memberModel, $login);

        // redirection vers login
        $urlLogin = $this->getUrlAskSucceed();
        return $response
            ->withStatus(302)
            ->withHeader('location', $urlLogin)
            ->withHeader('Cache-Control', 'private, no-cache, no-store');
    }

    /**
     * @param \Cms\Client\Member\Model\MemberRestfulModel $memberModel
     * @return MemberRestfulModel
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     */
    protected function handleMemberRecord(MemberRestfulModel $memberModel)
    {
        try {
            // sauvegarde du membre
            $this->getLoginService()->getMemberClient()->saveItem($memberModel);
            return $memberModel;
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot update user information with API',
                $e,
                MemberErrorCode::SAVE_MEMBER_API_FAILED
            );
        }
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Psr\Http\Message\ResponseInterface|mixed
     * @throws \InvalidArgumentException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\FieldException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     */
    public function activate(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        $member = $this->handleActivated($request);

        $email = $this->resolvEmail($member->email);
        $this->sendConfirmMail($email, $member);

        $response = $response
            ->withStatus(302)
            ->withHeader('location', $this->getUrlActivationSucceed())
            ->withHeader('Cache-Control', 'private, no-cache, no-store');
        return $response;
    }

    /**
     * Renvoi le conteneur du formulaire de creationn de compte
     * @param array $dataset
     * @param \Cms\Bundle\Member\Exception\RuntimeException|null $error
     * @return \Move\Http\Strategy\DataContainer\EngineDataContainer
     * @internal param null|string $errorCode
     */
    abstract protected function getEngineContainer(
        array $dataset = [],
        RuntimeException $error = null
    ) : EngineDataContainer;

    /**
     * Renvoi l'url de redirection apres demande d'activation du compte
     * @return string
     */
    abstract protected function getUrlAskSucceed() : string;

    /**
     * Renvoi l'url de redirection apres validation de l'activation
     * @return string
     */
    abstract protected function getUrlActivationSucceed() : string;

    /**
     * Envoi le mail d'activation du compte
     * @param string $email
     * @param \Cms\Client\Member\Model\MemberRestfulModel $member
     * @param \Cms\Model\Member\MemberLoginModel $loginModel
     * @return bool
     */
    abstract protected function sendActivationMail(
        string $email,
        MemberRestfulModel $member,
        MemberLoginModel $loginModel
    ) : bool;

    /**
     * Envoi du mail de confirmation d'activation du compte
     * @param string $email
     * @param \Cms\Client\Member\Model\MemberRestfulModel $member
     * @return bool
     */
    abstract protected function sendConfirmMail(
        string $email,
        MemberRestfulModel $member
    ) : bool;
}
