<?php

namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\FieldMissingException;
use Cms\Bundle\Member\Exception\RuntimeException;
use Move\Http\RequestUtils;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class AbstractLoginController
 * @package Cms\Bundle\Member
 */
abstract class AbstractLoginController implements
    LoggerAwareInterface
{

    use LoggerAwareTrait;
    use MemberUtilsTrait;
    use MemberLoginTrait;

    /** @var \Cms\Bundle\Member\MemberLoginService */
    private $loginService;

    /**
     * AbstractLoginController constructor.
     * @param \Cms\Bundle\Member\MemberLoginService $loginService
     */
    public function __construct(MemberLoginService $loginService)
    {
        $this->loginService = $loginService;
    }

    /**
     * @return \Cms\Bundle\Member\MemberLoginService
     */
    public function getLoginService() : MemberLoginService
    {
        return $this->loginService;
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Psr\Http\Message\ResponseInterface|mixed
     * @throws \Cms\Bundle\Member\Exception\InvalidStateException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     * @throws \InvalidArgumentException
     */
    public function login(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        // Récupération des données
        $parsedBody = $request->getParsedBody();
        // On checke si on a bien un login et un mot de passe
        if (isset($parsedBody['login'], $parsedBody['password'])) {
            $remembered = $parsedBody['remembered'] ?? false;
            $response = $this->withLoginPasswd(
                $parsedBody['login'],
                $parsedBody['password'],
                $response,
                $remembered
            );
            $response = $response
                ->withStatus(302)
                ->withHeader('location', $this->getUrlLoginSucceed($request))
                ->withHeader('Cache-Control', 'private, no-cache, no-store');
            return $response;
        }

        throw new FieldMissingException('email or password missing');
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Psr\Http\Message\ResponseInterface|mixed
     * @throws \Cms\Bundle\Member\Exception\RuntimeException
     * @throws \InvalidArgumentException
     */
    public function logout(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        // récupération du cookie
        $cookie = RequestUtils::GetCookieValue($request, MemberHelper::COOKIE_MBRLOGIN);

        // check si le cookie existe
        if ($cookie) {
            $response = $this->withLogout($response);
            $response = $response
                ->withStatus(302)
                ->withHeader('location', $this->getUrlLogoutSucceed($request))
                ->withHeader('Cache-Control', 'private, no-cache, no-store');
            return $response;
        }

        throw new RuntimeException('no cookie found');
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return string
     */
    abstract protected function getUrlLoginSucceed(ServerRequestInterface $request) : string;

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return string
     */
    abstract protected function getUrlLogoutSucceed(ServerRequestInterface $request) : string;
}
