<?php


namespace Cms\Controller;

use Cms\Client\Article\Model\ArticleRestfulModel;
use Move\Template\Html\Meta\Property;
use Move\Template\MetaEngineTrait;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\RedirectResponse;

/**
 * Class AbstractArticleController
 * @package Cms\Controller
 */
trait ArticleControllerTrait
{

    use MetaEngineTrait;

    /**
     * @param ArticleRestfulModel $article
     * @param bool $forAmp
     * @return string
     */
    abstract protected function buildArticleUrl(ArticleRestfulModel $article, $forAmp = false);

    /**
     * @param ArticleRestfulModel $article
     * @param ServerRequestInterface $request
     * @return null|RedirectResponse
     */
    protected function getArticleRedirect(
        ArticleRestfulModel $article,
        ServerRequestInterface $request
    ) {
        // récuperation de l'url canonique
        $articleUrl = $this->buildArticleUrl($article);
        // si la canonique est différente de l'url appelée, on envoie une redir
        if ($articleUrl !== $request->getUri()->getPath() || !empty($request->getUri()->getQuery())) {
            return new RedirectResponse($articleUrl, 301);
        }
        return null;
    }

    /**
     * @param ArticleRestfulModel $article
     * @param string $assetUrl
     * @throws \InvalidArgumentException
     */
    protected function addCommonArticleMetas(ArticleRestfulModel $article, $assetUrl = null)
    {
        // generate URL
        $articleUrl = $this->buildArticleUrl($article);

        // Facebook Open Graph
        $this
            ->addMetas(new Property('og:type', 'article'))
            ->addMetas(new Property('og:url', $articleUrl))
            ->addMetas(new Property('og:article:author', $article->author_signature));

        // Article info
        $this
            ->addMetas(new Property(
                'article:published_time',
                $article->first_published_at->format(DATE_W3C)
            ))
            ->addMetas(new Property(
                'article:modified_time',
                $article->last_published_at->format(DATE_W3C)
            ));

        // ajoute l'image
        if (!empty($assetUrl)) {
            $this
                ->addMetas(new Property('og:image', $assetUrl))
                ->addMetas(new Property('twitter:image', $assetUrl));
        }

        // Google Actu
        $this->addMetas(new Property('news_keywords', $article->short_title));
    }
}
