<?php


namespace Cms\Model\Contract\Product;

use Cms\LinkResource\LinkResourceModel;
use Cms\Model\AbstractConfigModel;
use Cms\Model\Contract\Config\ConfigBonPlanModel;
use Cms\Model\Contract\Config\ConfigCoupDeCoeurModel;
use Cms\Model\Contract\Config\ConfigLienInternetModel;
use Cms\Model\Contract\Config\ConfigLienReservationModel;
use Cms\Model\Contract\Config\ConfigLienThematicLoisirModel;
use Cms\Model\Contract\Config\ConfigLienThematicTourismModel;
use Cms\Model\Contract\Config\ConfigLogoPartnerModel;
use Cms\Model\Contract\Config\ConfigSelectionModel;
use Cms\Model\Contract\Config\ConfigVideoModel;

/**
 * Class ProductFactory
 * @package Cms\Model\Contract\Product
 */
class ProductFactory
{

    /** @var array */
    private static $profilClassMap = [
        ProductProfile::LIEN_INTERNET => ConfigLienInternetModel::class,
        ProductProfile::LIEN_THEMATIC_LOISIR => ConfigLienThematicLoisirModel::class,
        ProductProfile::LIEN_THEMATIC_TOURISM => ConfigLienThematicTourismModel::class,
        ProductProfile::COUP_DE_COEUR => ConfigCoupDeCoeurModel::class,
        ProductProfile::BON_PLAN => ConfigBonPlanModel::class,
        ProductProfile::LOGO_PARTENAIRE => ConfigLogoPartnerModel::class,
        ProductProfile::SELECTION => ConfigSelectionModel::class,
        ProductProfile::LIEN_RESERVATION => ConfigLienReservationModel::class,
        ProductProfile::VIDEO => ConfigVideoModel::class,
    ];

    /**
     * @param string|\Cms\Model\Contract\Product\ProductProfile $profil
     * @param array $configData
     * @param int|null $contractId
     * @return \Cms\Model\Contract\Product\ProductModel
     * @throws \UnexpectedValueException
     * @throws \InvalidArgumentException
     */
    public static function createProduct(
        $profil,
        array $configData = [],
        int $contractId = null
    ) : ProductModel {
        // check du profil passé
        $profil = (string)$profil;
        if (!ProductProfile::isValid($profil)) {
            throw new \InvalidArgumentException('the profil ' . $profil . ' is invalid');
        }
        // creation du produit
        $product = new ProductModel();
        $product->profil = $profil;
        if ($contractId !== null) {
            $product->contract_id = $contractId;
        }
        // creation de la config
        $configModel = static::createConfigModel($profil, $configData);
        $product->setConfig($configModel);
        return $product;
    }

    /**
     * @param string|\Cms\Model\Contract\Product\ProductProfile $profil
     * @param array $configData
     * @return \Cms\Model\AbstractConfigModel|null
     * @throws \UnexpectedValueException
     */
    public static function createConfigModel(
        $profil,
        array $configData = []
    ) {
        $profil = (string)$profil;
        if (!isset(self::$profilClassMap[$profil])) {
            return null;
        }
        $config = new self::$profilClassMap[$profil]();
        $config->config_class = self::$profilClassMap[$profil];
        if (!$config instanceof AbstractConfigModel) {
            throw new \UnexpectedValueException('the config model must extend class ' . AbstractConfigModel::class);
        }
        return $configData ? $config->populate($configData, true) : $config;
    }

    /**
     * @param int $productId
     * @param string|\Cms\Model\Contract\Product\ProductProfile $profil
     * @param array $configData
     * @param \Cms\LinkResource\LinkResourceModel|null $resourceModel
     * @return \Cms\Model\Contract\Product\ProductConfigModel
     * @throws \UnexpectedValueException
     * @throws \InvalidArgumentException
     */
    public static function createConfigByProductId(
        int $productId,
        $profil,
        array $configData = [],
        LinkResourceModel $resourceModel = null
    ) : ProductConfigModel {
        // check du profil passé
        $profil = (string)$profil;
        if (!ProductProfile::isValid($profil)) {
            throw new \InvalidArgumentException('the profil ' . $profil . ' is invalid');
        }
        // creation de la conf
        $config = new ProductConfigModel();
        $config->product_id = $productId;

        // ajout des donnée de resource
        if ($resourceModel !== null) {
            $config->link_provider = $resourceModel->link_provider;
            $config->link_ref = $resourceModel->link_ref;
            $config->link_prefs = $resourceModel->link_prefs;
            $config->link_type = $resourceModel->link_type;
        }

        // creation de la config
        $configModel = static::createConfigModel($profil, $configData);
        $config->setConfig($configModel);

        return $config;
    }

    /**
     * @param \Cms\Model\Contract\Product\ProductModel $product
     * @param array $configData
     * @param \Cms\LinkResource\LinkResourceModel|null $resourceModel
     * @return \Cms\Model\Contract\Product\ProductConfigModel
     * @throws \UnexpectedValueException
     * @throws \InvalidArgumentException
     */
    public static function createConfigByProduct(
        ProductModel $product,
        array $configData = [],
        LinkResourceModel $resourceModel = null
    ) : ProductConfigModel {
        if (!$product->id) {
            throw new \InvalidArgumentException('the product must have an id');
        }
        return static::createConfigByProductId(
            $product->id,
            $product->profil,
            $configData,
            $resourceModel
        );
    }
}
