<?php

namespace Cms\Search\Address;

use Cms\Model\AddressBook\Address\AddressModel;
use Cms\Search\Mapping\AddressDocumentTrait;
use Cms\Search\Mapping\AddressPropertiesTrait;
use Cms\Search\Mapping\GeoAdminDocumentTrait;
use Cms\Search\Mapping\GeoAdminNestedMapping;
use Elastica\Document;
use Move\Search\Analyzer\SearchWordAnalyzer;
use Move\Search\Analyzer\WordAnalyzer;
use Move\Search\ElasticMappingBuilder;
use Move\Search\Mapping\ElasticMappingProperty;
use Move\Search\Mapping\KeywordMapping;
use Move\Search\Mapping\ObjectMapping;
use Move\Search\Mapping\TextMapping;
use Move\Utils\Str;

/**
 * Class AddressIndexFactory
 * @package Cms\Search\Address
 */
class AddressIndexFactory
{

    use AddressPropertiesTrait;
    use AddressDocumentTrait;
    use GeoAdminDocumentTrait;

    /** Nom du type pour les adresses dans l'index */
    const MAPPING_NAME = 'cms_address';

    /**
     * @return \Move\Search\ElasticMappingBuilder
     */
    public static function createMappingBuilder() : ElasticMappingBuilder
    {
        $attributes = [
            (new KeywordMapping('name')),
            new ElasticMappingProperty('quantity', 'long', false),
            (new TextMapping('value', false))
                ->withAnalyzer(new WordAnalyzer(), new SearchWordAnalyzer())
                ->withOption('fields', [
                    'raw' => [
                        'type' => 'keyword',
                        'index' => false,
                    ],
                ]),
        ];

        $props = array_merge(
            [
                (new ElasticMappingProperty('scope_id', 'long', false)),
                (new ElasticMappingProperty('category_id', 'long', false)),


                (new KeywordMapping('address_slug')),
                (new TextMapping('address_name'))
                    ->withAnalyzer(new WordAnalyzer(), new SearchWordAnalyzer())
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),
                (new TextMapping('address_firm', false))
                    ->withAnalyzer(new WordAnalyzer(), new SearchWordAnalyzer())
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),

                new ElasticMappingProperty('created_at', 'date'),
                new ElasticMappingProperty('modified_at', 'date'),

                new GeoAdminNestedMapping(),

                (new ObjectMapping('attributes', $attributes, false)),
                (new ObjectMapping('attributes_nested', $attributes, false))->isNested(true),
            ],
            (new self())->getAddressProperties()
        );
        $builder = new ElasticMappingBuilder(self::MAPPING_NAME, $props);
        return $builder;
    }

    /**
     * @param \Cms\Model\AddressBook\Address\AddressModel $addressModel
     * @param array $admins
     * @param array $categoriesIds
     * @param array $scopeIds
     * @param array $attributes
     * @return Document
     */
    public static function createDocument(
        AddressModel $addressModel,
        array $admins,
        array $categoriesIds,
        array $scopeIds,
        array $attributes = []
    ) : Document {
        $attributes = array_map(function ($item) {
            return [
                'name' => $item['name'],
                'value' => $item['value'],
                'quantity' => $item['quantity'],
            ];
        }, $attributes);

        // creation du dataset
        $dataset = self::buildAddressMapping('undefined', $addressModel);
        $dataset = array_merge($dataset, [
            'category_id' => $categoriesIds,
            'scope_id' => $scopeIds,

            'address_slug' => Str::toAscii($addressModel->name),
            'address_name' => $addressModel->name,
            'address_firm' => $addressModel->firm,

            'geo_admins_nested' => array_map(
                [self::class, 'buildGeoAdminMapping'],
                $admins
            ),

            'created_at' => $addressModel->created_at->format(DATE_W3C),
            'modified_at' => $addressModel->modified_at->format(DATE_W3C),

            'attributes' => $attributes,
            'attributes_nested' => $attributes,
        ]);

        // envoi des données
        $document = new Document($addressModel->id, $dataset, self::MAPPING_NAME);
        return $document;
    }
}
