<?php


namespace Move\Command;

use Move\Command\Command\CreateCommand;
use Move\Command\Command\DeleteCommand;
use Move\Command\Command\FetchCommand;
use Move\Command\Command\UpdateCommand;
use Move\Command\Exception\ConflictException;
use Move\Command\Exception\NotFoundException;
use Move\Command\Exception\RelationException;
use Move\Filter\Validator\StateValidatorInterface;
use POM\DomainObjectInterface;
use POM\MapperAbstract;

/**
 * Class AbstractCommandHandler
 * @package Move\Command
 */
abstract class AbstractCommandHandler
{

    public const CREATE_STRATEGY_INSERT = 'insert';

    public const CREATE_STRATEGY_REPLACE = 'replace';

    /**
     * @return MapperAbstract
     */
    abstract public function getMapper();

    /**
     * @param array $dataset
     * @return DomainObjectInterface
     */
    abstract public function handleData(array $dataset = []);


    /** @var  string */
    protected $createStrategy = self::CREATE_STRATEGY_INSERT;

    /** @var  StateValidatorInterface */
    protected $validator;

    /**
     * @param StateValidatorInterface $validator
     * @return $this
     */
    public function setValidator(StateValidatorInterface $validator)
    {
        $this->validator = $validator;
        return $this;
    }

    /**
     * @param CreateCommand $command
     * @return null|DomainObjectInterface
     * @throws \Exception
     */
    public function handleCreate(CreateCommand $command)
    {
        // inject les données
        $dataset = $command->getDataset();
        $object = $this->handleData($dataset);

        // valide le statut du membre
        if ($this->validator) {
            $this->validator->validateState($object);
        }

        // insert dans le repository
        try {
            if ($this->createStrategy === self::CREATE_STRATEGY_INSERT) {
                if ($this->getMapper()->insert($object)) {
                    return $object;
                }
            } elseif ($this->createStrategy === self::CREATE_STRATEGY_REPLACE) {
                if ($this->getMapper()->save($object)) {
                    return $object;
                }
            } else {
                throw new \UnexpectedValueException('createStrategy is invalid : ' . $this->createStrategy);
            }
        } catch (\PDOException $e) {
            throw $this->convertPDOException(
                $e,
                $this->getMapper()->getEntityTable(),
                $this->createStrategy
            );
        }

        return null;
    }

    /**
     * @param UpdateCommand $command
     * @return null|DomainObjectInterface
     * @throws \Exception
     */
    public function handleUpdate(UpdateCommand $command)
    {
        // récuperation de l'objet
        $object = $this->fetchById($command->getId());

        // inject des données
        $dataset = $command->getDataset();
        $object->populate($dataset);

        // valide le statut du membre
        if ($this->validator) {
            $this->validator->validateState($object);
        }

        // insert dans le repository
        try {
            if ($this->getMapper()->update($object)) {
                return $object;
            }
        } catch (\PDOException $e) {
            throw $this->convertPDOException(
                $e,
                $this->getMapper()->getEntityTable(),
                'update'
            );
        }

        return null;
    }

    /**
     * @param DeleteCommand $command
     * @return bool
     * @throws \Exception
     */
    public function handleDelete(DeleteCommand $command) : bool
    {
        try {
            if ($this->getMapper()->removeById($command->getId())) {
                return true;
            }
        } catch (\PDOException $e) {
            throw $this->convertPDOException($e, $this->getMapper()->getEntityTable(), 'delete');
        }
        return false;
    }

    /**
     * @param FetchCommand $command
     * @return DomainObjectInterface
     * @throws \Move\Command\Exception\NotFoundException
     */
    public function handleFetch(FetchCommand $command)
    {
        // récuperation de l'objet
        $object = $this->fetchById($command->getId());
        return $object;
    }

    /**
     * @param mixed $objectId
     * @return DomainObjectInterface
     * @throws \Move\Command\Exception\NotFoundException
     */
    protected function fetchById($objectId)
    {
        $object = $this->handleData();
        if (!$this->getMapper()->fetchById($objectId, $object)) {
            $message = vsprintf('object id: %s not found in table: %s', [
                $objectId,
                $this->getMapper()->getEntityTable(),
            ]);
            throw new NotFoundException($message);
        }
        return $object;
    }

    /**
     * @param \PDOException $e
     * @param null|string $field
     * @param null $action
     * @return RelationException|ConflictException|\RuntimeException
     */
    protected function convertPDOException(\PDOException $e, $field = null, $action = null)
    {
        $errorCode = !empty($e->errorInfo[1]) ? $e->errorInfo[1] : 0;
        $message = $field ?: (!empty($e->errorInfo[2]) ? $e->errorInfo[2] : null);
        if ($errorCode) {
            switch ($e->errorInfo[1]) {
                case 1452:
                    // erreur sur les clé etrangere
                    return new RelationException($message, $errorCode, $e);
                    break;
                case 1062:
                    // conflict
                    return new ConflictException($message, $errorCode, $e);
                    break;
            }
        }
        return new \RuntimeException(
            "$field $action failed on query",
            $errorCode,
            $e
        );
    }
}
