<?php


namespace Cms\Bundle\Member;

use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Model\Member\MemberLoginModel;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class AbstractLoginForgetController
 * @package Cms\Bundle\Member
 */
abstract class AbstractLoginForgetController implements LoggerAwareInterface
{

    use LoggerAwareTrait;
    use MemberLoginForgetTrait;

    /** @var \Cms\Bundle\Member\MemberLoginService */
    private $loginService;

    /**
     * AbstractLoginController constructor.
     * @param \Cms\Bundle\Member\MemberLoginService $loginService
     */
    public function __construct(MemberLoginService $loginService)
    {
        $this->loginService = $loginService;
    }

    /**
     * @return \Cms\Bundle\Member\MemberLoginService
     */
    public function getLoginService() : MemberLoginService
    {
        return $this->loginService;
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Psr\Http\Message\ResponseInterface|mixed
     * @throws \Cms\Bundle\Member\Exception\LoginNotFoundException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \InvalidArgumentException
     */
    public function askPasswd(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        // execution de la demande de nouveau mot de passe
        $member = $this->handleAskPasswd($request);

        $scopeId = $this->getLoginService()->getScopeId();
        $loginModel = $member->getLoginByScope($scopeId);

        $email = $this->resolvEmail($member->email);
        $this->sendNewPasswdMail($email, $member, $loginModel);

        // redirection si ok
        $response = $response
            ->withStatus(302)
            ->withHeader('location', $this->getUrlAskSucceed())
            ->withHeader('Cache-Control', 'private, no-cache, no-store');
        return $response;
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Psr\Http\Message\ResponseInterface|mixed
     * @throws \Cms\Bundle\Member\Exception\LoginNotFoundException
     * @throws \Cms\Bundle\Member\Exception\PasswordFailedException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     * @throws \Cms\Bundle\Member\Exception\FieldException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \InvalidArgumentException
     */
    public function changePasswd(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        $member = $this->handleChangePasswd($request);

        $email = $this->resolvEmail($member->email);
        $this->sendConfirmMail($email, $member);

        // redirection si ok
        $response = $response
            ->withStatus(302)
            ->withHeader('location', $this->getUrlChangeSucceed())
            ->withHeader('Cache-Control', 'private, no-cache, no-store');
        return $response;
    }

    /**
     * @return string
     */
    abstract protected function getUrlAskSucceed() : string;

    /**
     * @return string
     */
    abstract protected function getUrlChangeSucceed() : string;

    /**
     * @param string $email
     * @param \Cms\Client\Member\Model\MemberRestfulModel $member
     * @param \Cms\Model\Member\MemberLoginModel $loginModel
     * @return bool
     */
    abstract protected function sendNewPasswdMail(
        string $email,
        MemberRestfulModel $member,
        MemberLoginModel $loginModel
    ) : bool;

    /**
     * @param string $email
     * @param \Cms\Client\Member\Model\MemberRestfulModel $member
     * @return bool
     */
    abstract protected function sendConfirmMail(
        string $email,
        MemberRestfulModel $member
    ) : bool;
}
