<?php


namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\ApiFailedException;
use Cms\Bundle\Member\Exception\FieldException;
use Cms\Bundle\Member\Exception\FieldMissingException;
use Cms\Bundle\Member\Exception\LoginNotFoundException;
use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Model\Member\MemberLoginState;
use Cms\Model\Member\MemberUtils;
use Move\Http\RequestUtils;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Trait LoginForgetTrait
 * @package Cms\Bundle\Member
 */
trait MemberLoginForgetTrait
{

    use MemberPasswordTrait;
    use MemberUtilsTrait;

    /**
     * @return \Cms\Bundle\Member\MemberLoginService
     */
    abstract public function getLoginService() : MemberLoginService;

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return \Cms\Client\Member\Model\MemberRestfulModel
     * @throws \Cms\Bundle\Member\Exception\LoginNotFoundException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     */
    protected function handleAskPasswd(
        ServerRequestInterface $request
    ) : MemberRestfulModel {
        // Récupération de l'email
        $parsedBody = $request->getParsedBody();
        if (!isset($parsedBody['email'])) {
            throw new FieldMissingException('email');
        }

        // vérification d'email
        $memberModel = $this->getMemberModel(
            $this->getLoginService()->getMemberClient(),
            $parsedBody['email'],
            true
        );

        // verification du scope
        $scopeId = $this->getLoginService()->getScopeId();
        $login = $memberModel->getLoginByScope($scopeId);
        if (!$login) {
            throw new LoginNotFoundException($parsedBody['email'], $scopeId);
        }

        // envoi la demande de reset de passwd
        try {
            $memberModel = $this->getLoginService()->getMemberClient()->updateLoginMember(
                $memberModel->id,
                $scopeId,
                null,
                true
            );
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot save member info for email ' . $parsedBody['email'],
                $e,
                MemberErrorCode::SAVE_MEMBER_API_FAILED
            );
        }
        return $memberModel;
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return \Cms\Client\Member\Model\MemberRestfulModel
     * @throws \Cms\Bundle\Member\Exception\LoginNotFoundException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     * @throws \Cms\Bundle\Member\Exception\PasswordFailedException
     * @throws \Cms\Bundle\Member\Exception\FieldException
     * @throws \Exception
     */
    protected function handleChangePasswd(
        ServerRequestInterface $request
    ) : MemberRestfulModel {
        // Récupération des params
        $parsedBody = $request->getParsedBody();
        if (!isset($parsedBody['email'])) {
            throw new FieldMissingException('email');
        }
        if (!isset($parsedBody['rk'])) {
            throw new FieldMissingException('rk');
        }

        $this->validatePassword($parsedBody);

        // on récupère la clé et l'id qu'elle contient
        $idMember = MemberUtils::getMemberIdWithPasswordKey($parsedBody['rk']);
        if ($idMember === null) {
            throw new FieldException('rk', MemberErrorCode::KEY_INVALID);
        }

        // on récupère le member à partir de l'identifiant
        $memberModel = $this->getMemberModel(
            $this->getLoginService()->getMemberClient(),
            $idMember,
            true
        );

        // recuperation du login correspondant au scope
        $memberLogin = $memberModel->getLoginByScope(
            $scopeId = $this->getLoginService()->getScopeId()
        );
        if (empty($memberLogin)) {
            throw new LoginNotFoundException($idMember, $scopeId);
        }

        // on check si l'email et la clé envoyés correspondent au member récupéré
        $activeKey = $this->getQueryResetKey($request);
        if ($parsedBody['email'] !== $memberModel->email
            || $activeKey !== $memberLogin->reset_passwd_key
        ) {
            throw new FieldException('rk', MemberErrorCode::KEY_INVALID);
        }

        // on check si la clé est toujours valide
        if ($memberLogin->reset_passwd_used) {
            throw new FieldException('rk', MemberErrorCode::KEY_ALREADY_USED);
        }

        $dateLogin = clone $memberLogin->reset_passwd_at;
        $dateLogin->add(new \DateInterval('PT1H'));
        if ((string)$memberLogin['state'] !== MemberLoginState::WAITING) {
            $dateNow = new \DateTime();
            if ($memberLogin->reset_passwd_at > $dateNow
                || $dateNow > $dateLogin
            ) {
                throw new FieldException('rk', MemberErrorCode::KEY_EXPIRED);
            }
        }

        // si tout est OK, on met à jour le mot de passe et on modifie l'état de la key
        try {
            $memberModel = $this->getLoginService()->getMemberClient()
                ->updateLoginMember(
                    $memberModel->id,
                    $this->getLoginService()->getScopeId(),
                    $parsedBody['password']
                );
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot save member info for email ' . $memberLogin['email'],
                $e,
                MemberErrorCode::SAVE_MEMBER_API_FAILED
            );
        }
        return $memberModel;
    }


    /**
     * extrait de la request la reset key, si aucune key dans la request on renvoi null
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return string|null
     */
    protected function getQueryResetKey(ServerRequestInterface $request)
    {
        return RequestUtils::getQueryValue($request, 'rk');
    }

    /**
     * @param string $uri
     * @param string $resetKey
     * @return string
     */
    protected function decorateUri(string $uri, string $resetKey) : string
    {
        $uri .= '?rk=' . urlencode($resetKey);
        return $uri;
    }
}
