<?php


namespace Cms\LinkResource\Provider\CMS;

use Cms\Client\AddressBook\AddressBookAddressClient;
use Cms\Client\AddressBook\AddressBookClient;
use Cms\Client\AddressBook\AddressBookTripCategoryClient;
use Cms\Client\AddressBook\AddressBookTripClient;
use Cms\Client\Service\AddressBookService;
use Cms\LinkResource\LinkResourceModel;
use Cms\LinkResource\LinkResourceType;
use Cms\LinkResource\Provider\ProviderClientAdapterInterface;
use Cms\LinkResource\Provider\ProviderRestfulClientAdapter;
use Cms\LinkResource\Provider\SearchAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareTrait;
use Cms\Model\AddressBook\Address\AddressCategoryState;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Log\LoggerInterface;

/**
 * Class CmsAddressBookProvider
 * @package Cms\LinkResource\Provider\CMS
 */
class CmsAddressBookProvider extends AbstractCmsAddressBookProvider implements
    SearchAdapterAwareInterface
{

    use SearchAdapterAwareTrait;

    /**
     * @var AddressBookService
     */
    private $bookService;

    /**
     * CmsAddressBookProvider constructor.
     * @param AddressBookService $bookService
     * @param \Psr\Cache\CacheItemPoolInterface|null $cachePool
     * @param \Psr\Log\LoggerInterface|null $logger
     */
    public function __construct(
        AddressBookService $bookService,
        CacheItemPoolInterface $cachePool = null,
        LoggerInterface $logger = null
    ) {
        parent::__construct($cachePool, $logger);
        $this->bookService = $bookService;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @return AddressBookAddressClient|AddressBookClient|AddressBookTripClient|AddressBookTripCategoryClient|null
     */
    protected function getClientByTypeCode($typeCode)
    {
        switch ((string)$typeCode) {
            case LinkResourceType::ADDRESS_BOOK:
                return $this->bookService->getAddressBookClient();
                break;
            case LinkResourceType::ADDRESS_ADDRESS:
                return $this->bookService->getAddressBookAddressClient();
                break;
            case LinkResourceType::ADDRESS_TRIP:
                return $this->bookService->getAddressBookTripClient();
                break;
            case LinkResourceType::ADDRESS_TRIP_CATEGORY:
                return $this->bookService->getAddressBookTripCategoryClient();
                break;
        }
        return null;
    }

    /**
     * @param mixed $typeCode
     * @return \Cms\LinkResource\Provider\ProviderClientAdapterInterface
     * @throws \RuntimeException
     */
    protected function getClientAdapter($typeCode) : ProviderClientAdapterInterface
    {
        $client = $this->getClientByTypeCode($typeCode);
        if ($client !== null) {
            return new ProviderRestfulClientAdapter($client);
        }
        throw new \RuntimeException('no client adapter');
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param mixed $query
     * @return LinkResourceModel[]
     */
    public function execSearch($typeCode, $query)
    {
        if (is_numeric($query)) {
            $resource = $this->getData($typeCode, $query);
            if ($resource) {
                return [
                    $resource->id,
                ];
            }
        }
        switch ((string)$typeCode) {
            case LinkResourceType::ADDRESS_ADDRESS:
                $categories = $this->bookService->getAddressBookAddressCategoryClient()
                    ->getBuilder()
                    ->setState(AddressCategoryState::ONLINE)
                    ->setScope($this->scopeId)
                    ->getIndex(-1);
                $results = $this->getClientByTypeCode($typeCode)
                    ->fetchAllBySearch(array_map(function ($category) {
                        return $category['id'];
                    }, iterator_to_array($categories, false)), $query);
                break;

            case LinkResourceType::ADDRESS_TRIP:
            case LinkResourceType::ADDRESS_BOOK:
                $results = $this->getClientByTypeCode($typeCode)
                    ->fetchAllBySearch($this->scopeId, $query);
                break;
        }
        if (empty($results)) {
            return [];
        }
        // On récupère les id de chaque modele pour les renvoyer
        $results = iterator_to_array($results, false);
        foreach ($results as &$model) {
            $model = $model['id'];
        }
        return $results;
    }
}
