<?php


namespace Cms\Route;

use Cms\LinkResource\LinkResourceHandler;
use Cms\LinkResource\LinkResourceHandlerAwareInterface;
use Cms\LinkResource\LinkResourceHandlerAwareTrait;
use Cms\LinkResource\LinkResourceModel;
use League\Uri\Components\Query;
use function League\Uri\build;
use function League\Uri\parse;

/**
 * Class UrlBuilder
 * @package Cms\Route
 */
class UrlBuilder implements
    LinkResourceHandlerAwareInterface
{

    use LinkResourceHandlerAwareTrait;

    /** @var AbstractRoute[] */
    private $routes;

    /** @var array */
    private $alias;

    /** @var string */
    private $host;

    /** @var string */
    private $scheme;

    /**
     * UrlBuilder constructor.
     * @param LinkResourceHandler $resourceHandler
     * @param string $host
     * @param string $scheme
     */
    public function __construct(
        LinkResourceHandler $resourceHandler,
        string $host = null,
        string $scheme = null
    ) {
        $this->resourceHandler = $resourceHandler;
        $this->host = $host;
        $this->scheme = $scheme;
    }

    /**
     * @param string $identifier
     * @param AbstractRoute $route
     * @param string|array|null $alias
     * @return $this
     */
    public function addRoute($identifier, AbstractRoute $route, $alias = null)
    {
        $this->routes[$identifier] = $route;
        if (!empty($alias)) {
            $this->addRouteAlias($identifier, $alias);
        }
        return $this;
    }

    /**
     * @param string $identifier
     * @param string|array $alias
     * @return $this
     */
    public function addRouteAlias($identifier, $alias)
    {
        if (!empty($alias)) {
            if (!\is_array($alias)) {
                $alias = [$alias];
            } else {
                $alias = array_unique($alias);
            }
            foreach ($alias as $alia) {
                $this->alias[$alia] = $identifier;
            }
        }
        return $this;
    }

    /**
     * @param string $currentUrl
     * @param string|array $params
     * @return string
     * @throws \InvalidArgumentException
     */
    public function addQueryParams(string $currentUrl, $params) : string
    {
        if (empty($params)) {
            return $currentUrl;
        }
        if (!\is_string($params) && !\is_array($params)) {
            throw new \InvalidArgumentException('params must be array or string');
        }
        $queryMerge = \is_array($params)
            ? Query::createFromParams($params)
            : new Query($params);
        $components = parse($currentUrl);
        if (!empty($components['query'])) {
            $query = new Query($components['query']);
            $query->merge($queryMerge)->ksort();
            $components['query'] = $query->getContent();
        } else {
            $components['query'] = $queryMerge->ksort()->getContent();
        }
        return build($components);
    }

    /**
     * @param string $currentUrl
     * @param null|string $host
     * @param null|string $scheme
     * @return string
     */
    public function urlToAbsolute($currentUrl, $host = null, $scheme = null) : string
    {
        $host = $host ?: $this->host;
        $scheme = $scheme ?: $this->scheme;
        if (!filter_var($currentUrl, FILTER_VALIDATE_URL)) {
            if (!preg_match('@^/@', $currentUrl)) {
                $currentUrl = '/' . $currentUrl;
            }
            return ($scheme ? $scheme . ':' : '') . '//' . $host . $currentUrl;
        }
        return $currentUrl;
    }

    /**
     * @param string $grpIdentifier
     * @param string $identifier
     * @param array $params
     * @return string
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public function buildAbsoluteUrl($grpIdentifier = null, $identifier = null, array $params = []) : string
    {
        $url = $this->buildUrl($grpIdentifier, $identifier, $params);
        if (!empty($this->host)) {
            $url = ($this->scheme ? $this->scheme . ':' : '') . '//' . $this->host . $url;
        } else {
            trigger_error('hostname is not define to build abs_url', E_USER_WARNING);
        }
        return $url;
    }

    /**
     * @param string $groupName
     * @return \Cms\Route\AbstractRoute|null
     */
    public function getRouteForGroup(string $groupName)
    {
        if (isset($this->alias[$groupName])) {
            $groupName = $this->alias[$groupName];
        }
        return $this->routes[$groupName] ?? null;
    }

    /**
     * @param string $groupName
     * @param string|null $identifier
     * @param array|null $params
     * @return \Cms\Route\PathBuilder|null
     * @throws \InvalidArgumentException
     */
    public function getPathBuilder(
        string $groupName,
        string $identifier = null,
        array $params = null
    ) {
        $identifier = $identifier ?: 'default';
        // récuperation de la route
        $route = $this->getRouteForGroup($groupName);
        if ($route === null) {
            return null;
        }
        $pathBuilder = $route->getPathBuilder($identifier);
        // merge des données
        if ($params !== null) {
            $pathBuilder->mergeParams($params);
        }
        return $pathBuilder;
    }

    /**
     * @param \Cms\Route\AbstractRoute $route
     * @param string $identifier
     * @param array|null $params
     * @return string|null
     * @throws \InvalidArgumentException
     */
    public function buildWithRoute(
        AbstractRoute $route,
        string $identifier = null,
        array $params = null
    ) {
        $identifier = $identifier ?: 'default';
        // merge des données
        if ($params !== null) {
            $route
                ->getPathBuilder($identifier)
                ->mergeParams($params);
        }
        return $route->buildPath($identifier);
    }

    /**
     * @param string $groupName
     * @param string $identifier
     * @param array|null $params
     * @return callable
     * @throws \InvalidArgumentException
     */
    public function prepareBuild(
        string $groupName,
        string $identifier = null,
        array $params = null
    ) : callable {
        $identifier = $identifier ?: 'default';
        // récuperation de la route
        $route = $this->getRouteForGroup($groupName);
        if ($route === null) {
            // fonction vide
            return function () {
            };
        }
        // merge des données
        if ($params !== null) {
            $route
                ->getPathBuilder($identifier)
                ->mergeParams($params);
        }
        return $route->prepareBuilder($identifier);
    }

    /**
     * @param \Cms\LinkResource\LinkResourceModel $resourceModel
     * @param string|null $identifier
     * @param array|null $params
     * @return string|null
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public function buildWithResource(
        LinkResourceModel $resourceModel,
        string $identifier = null,
        array $params = null
    ) {
        try {
            $resourceData = $this
                ->getLinkResourceHandler()
                ->getDataByModel($resourceModel);
        } catch (\Exception $e) {
            throw new \RuntimeException(
                'getting data model with resourceHandler failed',
                null,
                $e
            );
        }
        $prefIdentifier = $resourceModel->link_prefs['identifier'] ?? 'default';
        $identifier = $identifier ?: $prefIdentifier;
        $params['model'] = $resourceData;
        $route = $this->getRouteForGroup((string)$resourceModel->link_type);
        if ($route === null) {
            return $route;
        }
        return $route->buildWithArray((string)$identifier, $params);
    }

    /**
     * @param string $grpIdentifier
     * @param string $identifier
     * @param array|null $params
     * @return string|null
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public function buildUrl($grpIdentifier, $identifier = null, $params = null)
    {
        // si on passe un linkresourceModel
        if ($grpIdentifier instanceof LinkResourceModel) {
            return $this->buildWithResource($grpIdentifier);
        }
        // check du type pour compatibilité
        if (!\is_array($params) && $params !== null) {
            $params = [];
        }
        $route = $this->getRouteForGroup((string)$grpIdentifier ?: 'default');
        if ($route === null) {
            return $route;
        }
        return $route->buildWithArray((string)$identifier, $params);
    }
}
