<?php

namespace Move\Restful;

use Move\Http\Client\Exception\HttpClientException;
use Move\Http\Client\Exception\HttpRequestException;
use Move\Http\Client\HttpClientInterface;
use Move\Http\Client\HttpRequestOptions;
use Move\Http\Middleware\BodyParser;
use Move\Iterator\MapIterator;
use Move\ObjectMapper\ObjectTransformer;
use Move\Utils\Arr;
use POM\DomainObjectInterface;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LogLevel;
use Psr\Log\NullLogger;

/**
 * Class AbstractRestfulClient
 * @package Move\Restful
 */
abstract class AbstractRestfulClient implements LoggerAwareInterface
{

    use LoggerAwareTrait;


    public const UPDATE_EXCEPT_EMPTY = 100;

    public const UPDATE_EXCEPT_NULL = 200;

    public const UPDATE_EXCEPT_NONE = 300;


    public const CREATE_EXCEPT_EMPTY = 100;

    public const CREATE_EXCEPT_NULL = 200;

    public const CREATE_EXCEPT_NONE = 300;


    /**
     * @var int
     */
    protected $updatePolicy;

    /**
     * @var int
     */
    protected $createPolicy;

    /**
     * @var HttpClientInterface
     */
    private $client;

    /**
     * @var ObjectTransformer
     */
    private $transformer;

    /**
     * AbstractRestfulClient constructor.
     * @param HttpClientInterface $client
     * @param callable $transformer
     */
    public function __construct(HttpClientInterface $client, callable $transformer = null)
    {
        $this->client = $client;
        $this->transformer = $transformer ?: new ObjectTransformer();
        $this->logger = new NullLogger();
        $this->updatePolicy = self::UPDATE_EXCEPT_EMPTY;
        $this->createPolicy = self::CREATE_EXCEPT_EMPTY;
    }

    /**
     * @return ObjectTransformer
     */
    public function getTransformer()
    {
        return $this->transformer;
    }

    /**
     * @return HttpClientInterface
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * @param int $updatePolicy
     * @return $this
     */
    public function setUpdatePolicy($updatePolicy)
    {
        $this->updatePolicy = $updatePolicy;
        return $this;
    }

    /**
     * @param int $createPolicy
     * @return $this
     */
    public function setCreatePolicy($createPolicy)
    {
        $this->updatePolicy = $createPolicy;
        return $this;
    }

    /**
     * @param string $method
     * @param array $segments
     * @param array $options
     * @return array
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws HttpRequestException
     */
    final protected function request($method, array $segments, $options = [])
    {
        // envoi de la requete
        try {
            $response = $this->getClient()->request($method, $segments, $options);
        } catch (HttpClientException $e) {
            return $this->handleHttpClientException($e);
        } catch (HttpRequestException $e) {
            return $this->handleHttpRequestException($e);
        }

        try {
            return $this->getDatasetFromResponse($response);
        } catch (\Exception $parseError) {
            throw new \LogicException(
                $this->getLogMessage(
                    'ParseBodyError',
                    vsprintf('request %s %s', [
                        $method,
                        implode('/', $segments),
                    ])
                ),
                0,
                $parseError
            );
        }
    }

    /**
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return array
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    protected function getDatasetFromResponse(ResponseInterface $response)
    {
        // retour vide
        if ($response->getStatusCode() === 204
            || $response->getBody()->getSize() === 0
        ) {
            return [];
        }

        // parsing du message
        $dataset = (new BodyParser())->parseMessage($response);
        return $dataset;
    }

    /** @var string */
    private $errorType = 'RestClientError';

    /**
     * @param string $identifier
     * @param string $msg
     * @return string
     */
    private function getLogMessage(string $identifier, string $msg) : string
    {
        $type = $this->errorType;
        $msg = $type . ' ' . $identifier . ' "' . $msg . '"';
        return $msg;
    }

    /**
     * @param \Exception $e
     * @param mixed $level
     * @param string $identifier
     * @param \Psr\Http\Message\RequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface|null $response
     */
    private function logException(
        \Exception $e,
        $level,
        string $identifier,
        RequestInterface $request,
        ResponseInterface $response = null
    ) {
        $type = $this->errorType;
        $msg = $this->getLogMessage($identifier, $e->getMessage());
        $ctx = [
            'request_uri' => (string)$request->getUri(),
            'request_meth' => $request->getMethod(),
            'err_message' => $e->getMessage(),
            'err_class' => \get_class($e),
            'err_type' => $type,
            'err_ident' => $identifier,
            'exception' => $e,
        ];
        if ($response !== null) {
            $ctx = array_merge($ctx, [
                'response_code' => $response->getStatusCode(),
                'response_reason' => $response->getReasonPhrase(),
            ]);
        }
        $this->logger->log($level, $msg, $ctx);
    }

    /**
     * @param \Move\Http\Client\Exception\HttpClientException $e
     * @return array
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     */
    protected function handleHttpClientException(HttpClientException $e)
    {
        // dans le cas de method GET si 404 on renvoi du vide
        if ($e->hasResponse()
            && (int)$e->getResponse()->getStatusCode() === 404
            && strtoupper($e->getRequest()->getMethod()) === 'GET'
        ) {
            $this->logException(
                $e,
                LogLevel::DEBUG,
                'NotFound',
                $e->getRequest(),
                $e->getResponse()
            );
            return [];
        }
        return $this->handleHttpRequestException($e);
    }

    /**
     * @param \Move\Http\Client\Exception\HttpRequestException $e
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \Move\Http\Client\Exception\HttpServerException
     */
    protected function handleHttpRequestException(HttpRequestException $e)
    {
        if ($e->hasResponse()) {
            // tentative de decodage de la reponse de l'erreur
            try {
                $dataset = (new BodyParser())->parseMessage($e->getResponse());
            } catch (\Exception $parseError) {
                $this->logException(
                    $parseError,
                    LogLevel::DEBUG,
                    'ParseError',
                    $e->getRequest(),
                    $e->getResponse()
                );
            }
            // si un message est transmis via le body on change l'exception
            if (isset($dataset['message'])) {
                throw HttpRequestException::create(
                    $e->getRequest(),
                    $e->getResponse(),
                    $dataset['message'],
                    $e
                );
            }
            // cas des 503 (varnish), on va les logger de lanière différente
            if ((int)$e->getResponse()->getStatusCode() === 503) {
                $this->logException(
                    $e,
                    LogLevel::WARNING,
                    'Proxy503Error',
                    $e->getRequest(),
                    $e->getResponse()
                );
            }
        }
        throw $e;
    }

    /**
     * @param array $segments
     * @param array $queryParams
     * @return \Iterator
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws HttpClientException
     */
    protected function loadIndexFromClient($segments, array $queryParams = [])
    {
        $dataset = $this->request('GET', [$segments], [
            HttpRequestOptions::QUERY_PARAMS => $queryParams,
        ]);
        if (!empty($dataset['data'])) {
            $result = new \ArrayIterator($dataset['data']);
            $result = new MapIterator($result, [$this, 'callbackIterator']);
            $result = new \CallbackFilterIterator($result, function ($value) {
                return null !== $value;
            });
            return $result;
        }
        return new \ArrayIterator();
    }

    /**
     * @param DomainObjectInterface $object
     * @param string $segment
     * @param int $id
     * @param array $queryParams
     * @return DomainObjectInterface|null
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     */
    protected function loadObjectFromClient(
        DomainObjectInterface $object,
        $segment,
        $id,
        array $queryParams = []
    ) {
        $options = [
            HttpRequestOptions::QUERY_PARAMS => $queryParams,
        ];
        $dataset = $this->request('GET', [$segment, $id], $options);
        if (empty($dataset)) {
            return null;
        }
        // chargement des donnée
        $object = $this->withJsonDataset($object, $dataset);
        return $object;
    }

    /**
     * @param DomainObjectInterface $object
     * @param string $segment
     * @param string $attrId
     * @param array $queryParams
     * @return null|\POM\DomainObjectInterface
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     */
    protected function updateObjectFromClient(
        DomainObjectInterface $object,
        $segment,
        $attrId = 'id',
        array $queryParams = []
    ) {
        $options = [
            HttpRequestOptions::QUERY_PARAMS => $queryParams,
            HttpRequestOptions::BODY_CONTENT => $this->convertObjectToArray(
                $object,
                $this->updatePolicy
            ),
        ];
        // lance la requete vers le server
        $jsonDataset = $this->request(
            'PUT',
            [$segment, $object[$attrId]],
            $options
        );
        if (\is_array($jsonDataset) && !empty($jsonDataset)) {
            $object = $this->withJsonDataset($object, $jsonDataset);
        }
        return $object;
    }

    /**
     * @param DomainObjectInterface $object
     * @param string $segment
     * @param array $mergeData
     * @param array $queryParams
     * @return bool|\POM\DomainObjectInterface
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     */
    protected function createObjectFromClient(
        DomainObjectInterface $object,
        $segment,
        array $mergeData = [],
        array $queryParams = []
    ) {
        $options = [
            HttpRequestOptions::QUERY_PARAMS => $queryParams,
            HttpRequestOptions::BODY_CONTENT => array_merge(
                $this->convertObjectToArray(
                    $object,
                    $this->createPolicy
                ),
                $mergeData
            ),
        ];
        // lance la requete vers le server
        $jsonDataset = $this->request(
            'POST',
            [$segment],
            $options
        );
        if (\is_array($jsonDataset) && !empty($jsonDataset)) {
            $object = $this->withJsonDataset($object, $jsonDataset);
        }
        return $object;
    }

    /**
     * @param DomainObjectInterface $object
     * @param string $segment
     * @param string $attrId
     * @param array $queryParams
     * @return array|bool|\POM\DomainObjectInterface
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     */
    protected function deleteObjectFromClient(
        DomainObjectInterface $object,
        $segment,
        $attrId = 'id',
        array $queryParams = []
    ) {
        $options = [
            HttpRequestOptions::QUERY_PARAMS => $queryParams,
        ];
        // lance la requete vers le server
        $jsonDataset = $this->request(
            'DELETE',
            [$segment, $object[$attrId]],
            $options
        );
        return $jsonDataset;
    }

    /**
     * @param DomainObjectInterface $object
     * @param array $jsonDataset
     * @return DomainObjectInterface
     */
    protected function withJsonDataset(DomainObjectInterface $object, array $jsonDataset)
    {
        // conversion des data
        $dataset = $this->convertApiArray($jsonDataset);
        // populate object
        $object = clone $object;
        $object->populate($dataset, true);
        return $object;
    }

    /**
     * @param DomainObjectInterface $object
     * @param null|int $filterPolicy
     * @return array
     */
    protected function convertObjectToArray(DomainObjectInterface $object, $filterPolicy = null)
    {
        return Arr::array_filter(
            $this->transformer->transform($object),
            function ($value) use ($filterPolicy) {
                switch ($filterPolicy) {
                    // on supprime rien
                    case self::UPDATE_EXCEPT_NONE:
                    case self::CREATE_EXCEPT_NONE:
                        return true;
                        break;

                    // on ne supprime que les valeur egale a null (not set)
                    case self::UPDATE_EXCEPT_NULL:
                    case self::CREATE_EXCEPT_NULL:
                        return $value !== null;
                        break;

                    // on ne supprime que les valeur vide
                    default:
                    case self::UPDATE_EXCEPT_EMPTY:
                    case self::CREATE_EXCEPT_EMPTY:
                        return !empty($value);
                        break;
                }
            }
        );
    }

    /**
     * @param array $jsonDataset
     * @return array
     */
    protected function convertApiArray(array $jsonDataset)
    {
        return $jsonDataset;
    }

    /**
     * @param array $dataset
     * @return DomainObjectInterface
     */
    abstract public function callbackIterator($dataset);
}
