<?php

namespace Move\Template\Extensions;

use League\Plates\Engine;
use League\Plates\Extension\ExtensionInterface;

/**
 * Class SmartAd
 * @package Move\Template\Extensions
 */
class SmartAd implements ExtensionInterface
{

    public const RENDER_DEFAULT = 0;

    public const RENDER_READY = 1;

    public const RENDER_ON_DEMAND = 2;

    public const RENDER_ASYNC = 3;

    /**
     * @var int
     */
    private static $SMART_PAGE_ID;

    /**
     * @var array
     */
    private static $TARGETS = [];

    /**
     * @var int
     */
    private $siteId;

    /**
     * @var string
     */
    private $host;

    /**
     * @var int
     */
    private $networkId;

    /**
     * @var int
     */
    private $renderMode;

    /**
     * SmartAd constructor.
     * @param int $siteId
     * @param string $host
     * @param $networkId
     * @param int $renderMode
     */
    public function __construct($siteId, $host, $networkId, $renderMode = null)
    {
        $this->siteId = (int)$siteId;
        $this->host = $host;
        $this->networkId = (int)$networkId;
        $this->renderMode = $renderMode ?: self::RENDER_DEFAULT;
    }

    /**
     * @param int $siteId
     * @param string $host
     * @param int $networkId
     * @return \Move\Template\Extensions\SmartAd
     */
    public static function createAsync(int $siteId, string $host, int $networkId) : SmartAd
    {
        $self = new self($siteId, $host, $networkId, self::RENDER_ASYNC);
        return $self;
    }

    /**
     * @param Engine $engine
     */
    public function register(Engine $engine)
    {
        $engine->registerFunction('sas_call', [$this, 'runCall']);
        $engine->registerFunction('sas_call_mobile_only', [$this, 'runCallMobileOnly']);
        $engine->registerFunction('sas_amp_call', [$this, 'runAmpCall']);
        $engine->registerFunction('sas_setup', [$this, 'runSetup']);
        $engine->registerFunction('sas_amp_setup', [$this, 'runAmpSetup']);
    }

    /**
     * @return string
     */
    public function runAmpSetup()
    {
        $code = <<<EOF
<script async custom-element="amp-ad" src="https://cdn.ampproject.org/v0/amp-ad-0.1.js"></script>
EOF;
        return $code;
    }

    /**
     * @param string $formatId
     * @param int $width
     * @param int $height
     * @return string
     */
    public function runAmpCall($formatId, $width, $height)
    {
        $pageId = self::$SMART_PAGE_ID;
        if (empty($pageId)) {
            return '<!--# sas_pageID missed #-->';
        }

        // rendu des targets
        $targets = [];
        foreach (self::$TARGETS as $targetName => $targetValues) {
            $targets[] = $targetName . '=' . implode(',', $targetValues);
        }
        $targets = implode(';', $targets);

        $siteId = $this->siteId;
        $host = $this->host;

        $code = <<<EOF
<amp-ad width=$width height=$height
     type="smartadserver"
     data-call="std"
     data-site="$siteId"
     data-page="$pageId"
     data-format="$formatId"
     data-target="$targets"
     data-domain="https://$host">
</amp-ad>
EOF;
        return $code;
    }

    /**
     * @param int $formatId
     * @return string
     */
    public function runCall($formatId)
    {
        $pageId = self::$SMART_PAGE_ID;
        if (empty($pageId)) {
            return '<!--# sas_pageID missed #-->';
        }
        $siteId = $this->siteId;
        $host = $this->host;
        $nwid = $this->networkId;
        $stamp = time();

        $jscript = <<<EOF
sas.call("std", {
    siteId:_sas_siteId,
    pageId:_sas_pageId,
    formatId:$formatId,
    target:_sas_target
});
EOF;

        // mode asynchrone
        $preScript = '';
        if ($this->renderMode === self::RENDER_ASYNC) {
            $preScript = '<div id="sas_' . $formatId . '"></div>';
            $jscript = <<<EOF
sas.cmd.push(function() {
    $jscript
});
EOF;
        }

        $code = <<<EOF
    $preScript
    <script>
        $jscript
    </script>
    <noscript>
        <a href="//$host/ac?jump=1&nwid=$nwid&siteid=$siteId&pgname=$pageId&fmtid=$formatId&visit=m&tmstp=[$stamp]&out=nonrich" target="_blank">
            <img src="//$host/ac?out=nonrich&nwid=$nwid&siteid=$siteId&pgname=$pageId&fmtid=$formatId&visit=m&tmstp=[$stamp]" border="0" alt="" /></a>
    </noscript>
EOF;

        return $code;
    }

    /**
     * @param int $formatId
     * @return string
     */
    public function runCallMobileOnly($formatId)
    {
        $pageId = self::$SMART_PAGE_ID;
        if (empty($pageId)) {
            return '<!--# sas_pageID missed #-->';
        }
        $siteId = $this->siteId;
        $host = $this->host;
        $nwid = $this->networkId;
        $stamp = time();

        $jscript = <<<EOF
if (window.innerWidth < 768) {
    sas.call("std", {
        siteId:_sas_siteId,pageId:_sas_pageId,formatId:$formatId,target:_sas_target
    });
}
EOF;

        // mode asynchrone
        $preScript = '';
        if ($this->renderMode === self::RENDER_ASYNC) {
            $preScript = '<div id="sas_' . $formatId . '"></div>';
            $jscript = <<<EOF
sas.cmd.push(function() {
    $jscript
});
EOF;
        }

        $code = <<<EOF
    $preScript
    <script>
        $jscript
    </script>
    <noscript>
        <a href="//$host/ac?jump=1&nwid=$nwid&siteid=$siteId&pgname=$pageId&fmtid=$formatId&visit=m&tmstp=[$stamp]&out=nonrich" target="_blank">
            <img src="//$host/ac?out=nonrich&nwid=$nwid&siteid=$siteId&pgname=$pageId&fmtid=$formatId&visit=m&tmstp=[$stamp]" border="0" alt="" /></a>
    </noscript>
EOF;

        return $code;
    }

    /**
     * @return string
     */
    public function runSetup()
    {
        $pageId = self::$SMART_PAGE_ID;
        if (empty($pageId)) {
            return '<!--# sas_pageID missed #-->';
        }
        $siteId = $this->siteId;
        $host = $this->host;
        $nwid = $this->networkId;
        $render = $this->renderMode;

        // rendu des targets
        $targets = [];
        foreach (self::$TARGETS as $targetName => $targetValues) {
            $targets[] = $targetName . '=' . implode(',', $targetValues);
        }
        $targets = implode(';', $targets);

        if ($render === self::RENDER_ASYNC) {
            $code = <<<EOF
<script src="//ced.sascdn.com/tag/$nwid/smart.js" async></script>
<script>
    var _sas_siteId = $siteId,
        _sas_pageId = $pageId,
        _sas_target = '$targets';
    var sas = sas || {};
    sas.cmd = sas.cmd || [];
    sas.cmd.push(function() {
        sas.setup({ networkid: $nwid, domain: "//$host", async: true });
    });
</script>
EOF;
        } else {
            $code = <<<EOF
    <script src='//$host/config.js?nwid=$nwid'></script>
    <script>
        var _sas_siteId = $siteId,
            _sas_pageId = $pageId,
            _sas_target = '$targets';
        sas.setup({ domain: '//$host', async: true, renderMode: $render});
    </script>
EOF;
        }
        return $code;
    }

    /**
     * @param string $targetName
     * @param string|array $targetValue
     */
    public static function addInTarget($targetName, $targetValue)
    {
        if (!isset(self::$TARGETS[$targetName])) {
            self::$TARGETS[$targetName] = [];
        }
        self::$TARGETS[$targetName][] = $targetValue;
    }

    /**
     * @param int $pageId
     */
    public static function setPageId($pageId)
    {
        self::$SMART_PAGE_ID = $pageId;
    }
}
