<?php

namespace Cms\Bundle\Sitemap;

use Cms\Client\Article\Model\CategoryRestfulModel;
use Cms\Client\Command\Article\Command\FetchAllArticleCategoryByFilterCommand;
use Cms\Client\Service\ArticleService;
use Cms\Client\Service\ArticleServiceAwareTrait;
use Cms\Model\Article\ArticleState;
use Cms\Route\AbstractRoute;
use Move\Iterator\MapIterator;
use function Move\I18n\callInLocale;
use function Move\I18n\localeCodeToLng;

/**
 * Class GenerateSitemapArticleHandler
 * @package Cms\Bundle\Sitemap
 */
class GenerateSitemapArticleHandler extends AbstractGenerateSitemapHandler
{

    use ArticleServiceAwareTrait;

    /** @var \Cms\Bundle\Sitemap\SitemapWriter */
    protected $writer;

    /**
     * GenerateSitemapArticleHandler constructor.
     * @param \Cms\Client\Service\ArticleService $articleService
     */
    public function __construct(ArticleService $articleService)
    {
        $this->articleService = $articleService;
        $this->writer = new SitemapWriter([], SitemapStruct::SITEMAP_HEADER_IMAGE);
    }

    /**
     * @param \Cms\Bundle\Sitemap\GenerateSitemapArticleCommand $command
     * @throws \DomainException
     */
    public function handleGenerateSitemapEdito(GenerateSitemapArticleCommand $command)
    {
        if (!\defined('SITEMAPS_PATH')) {
            throw new \DomainException('SITEMAPS_PATH constant is undefined');
        }

        $cmd = new FetchAllArticleCategoryByFilterCommand();
        $cmd->setScopeId($command->getScopeId());
        $categories = $this->articleService->getCategoryCommandBus()->handle($cmd);
        foreach ($categories as $category) {
            if ($category instanceof CategoryRestfulModel && !$category->anteriority) {
                $this->generateForCategoryId($category->id);
            }
        }

        // fermeture des fichier
        foreach ($this->getWriter()->close() as $locale => $tmpName) {
            $filename = SITEMAPS_PATH . '/edito-index-' . $locale . '.xml';
            rename($tmpName, $filename);
            chmod($filename, 0644);
        }
    }

    /**
     * @return \Cms\Route\AbstractRoute|null
     * @throws \DomainException
     */
    protected function getArticleRouteForGroup()
    {
        $urlBuilder = $this->getUrlBuilder();
        $articleRoute = $urlBuilder->getRouteForGroup('edito');
        return $articleRoute;
    }

    /**
     * @return \Cms\Route\AbstractRoute|null
     * @throws \DomainException
     */
    protected function getAssetRouteForGroup()
    {
        $urlBuilder = $this->getUrlBuilder();
        $assetRoute = $urlBuilder->getRouteForGroup('asset');
        return $assetRoute;
    }

    /**
     * @param \Cms\Route\AbstractRoute $articleRoute
     * @param array $article
     * @return null|string
     */
    protected function buildArticlePath(AbstractRoute $articleRoute, array $article)
    {
        $url = $articleRoute->buildWithArray('article', [
            'category' => $article['category']['slug'],
            'id' => $article['id'],
            'article' => $article['slug'],
        ]);
        return $url;
    }

    /**
     * @param \Cms\Route\AbstractRoute $assetRoute
     * @param array $assetData
     * @return null|string
     */
    protected function buildImagePath(AbstractRoute $assetRoute, array $assetData)
    {
        $assetUrl = $assetRoute->buildWithArray('full', array_merge($assetData, [
            'width' => 964,
            'height' => 603,
        ]));
        return $assetUrl;
    }

    /**
     * @param array $article
     * @param resource $fh
     * @throws \UnexpectedValueException
     * @throws \DomainException
     * @throws \InvalidArgumentException
     */
    protected function writeArticleLine($article, $fh)
    {
        // creation de la route (avec lang)
        $articleRoute = $this->getArticleRouteForGroup();
        if ($articleRoute === null) {
            throw new \UnexpectedValueException('article route is undefined');
        }
        $assetRoute = $this->getAssetRouteForGroup();
        if ($assetRoute === null) {
            throw new \UnexpectedValueException('asset route is undefined');
        }

        // récuperation des données article
        $images = [];
        if (!empty($article['assets'])) {
            foreach ($article['assets'] as $assetData) {
                $imageUrl = $this->buildImagePath($assetRoute, $assetData);
                $images[] = $this->getUrlBuilder()->urlToAbsolute($imageUrl);
            }
        }

        // données d'articles
        $lastMod = date(DATE_W3C, strtotime($article['modified_at']));
        $articleUrl = $this->buildArticlePath($articleRoute, $article);
        $articleUrl = $this->getUrlBuilder()->urlToAbsolute($articleUrl);

        // ecriture du fichier
        SitemapWriter::writeloc($fh, $articleUrl, $lastMod, $images);
    }

    /**
     * @param \Iterator $articles
     * @param resource $fh
     * @param string $locale
     */
    protected function generateArticleList($articles, $fh, string $locale)
    {
        foreach ($articles as $article) {
            callInLocale([$this, 'writeArticleLine'], $locale, [$article, $fh]);
        }
    }

    /**
     * @param int $categoryId
     * @throws \DomainException
     */
    protected function generateForCategoryId($categoryId)
    {
        // lecture des categories
        $categoryClient = $this->articleService->getCategoryClient();
        $categories = $categoryClient->getIndexByParentId($categoryId);
        $categories = new MapIterator($categories, $categoryClient->getTransformer());
        $categories = iterator_to_array($categories);
        $categoriesIds = [$categoryId];
        foreach ($categories as $category) {
            $categoriesIds[] = $category['id'];
        }

        // ouverture du pointeur de fichier
        foreach ($this->getWriter()->open() as $locale => $fh) {
            $articleClient = $this->articleService->getArticleClient();
            $builder = $articleClient->getBuilder()
                ->setState(ArticleState::PUBLISH)
                ->setLang(localeCodeToLng($locale))
                ->setCategories($categoriesIds);
            $page = $builder->getIndex(0);
            $this->logger->info(
                'index ' . $page->getTotal() . ' article for category ' . $categoryId,
                ['query' => $builder->getQueryParams(),]
            );

            $nbPage = ceil($page->getTotal() / $page->getByPage());
            for ($i = 0; $i < $nbPage; $i++) {
                $page = $builder->getIndex($i);
                $articles = new MapIterator($page, $this->articleService->getArticleTransformer());
                $this->generateArticleList($articles, $fh[0], $locale);
            }
        }
    }

    /**
     * @return \Cms\Bundle\Sitemap\SitemapWriter
     */
    protected function getWriter() : SitemapWriter
    {
        return $this->writer;
    }
}
