<?php


namespace Cms\Client\Article\Transformer;

use Cms\Client\Article\CategoryClient;
use Cms\Client\Article\Model\ArticleRestfulModel;
use Cms\DashParser\DashParser;
use Cms\DashParser\DashParserAwareTrait;
use Cms\DashParser\Serializer\HtmlBlockConverter;
use Cms\DashParser\Serializer\HtmlEntityConverter;
use Cms\DashParser\Serializer\HtmlSerializer;
use Cms\DashParser\Serializer\HtmlStyleConverter;
use Cms\Model\Article\Asset\AssetCoverType;
use Cms\Model\Article\Category\CategoryModel;
use Cms\Model\I18nHelper;
use Move\Http\Client\Exception\HttpRequestException;
use Move\I18n\I18nLanguageCode;
use Move\ObjectMapper\ObjectTransformer;
use POM\DomainObjectInterface;

/**
 * Class ArticleTransformer
 * @package Cms\Client\Article\Transformer
 */
class ArticleTransformer extends ObjectTransformer
{

    use DashParserAwareTrait;

    /** @var  \Cms\Client\Article\Model\CategoryRestfulModel[] */
    private static $categoryStore;

    /** @var  null|string */
    private $language;

    /** @var CategoryClient */
    protected $categoryClient;

    /** @var  HtmlStyleConverter */
    protected $htmlStyleConverter;

    /** @var  HtmlBlockConverter */
    protected $htmlBlockConverter;

    /** @var  HtmlEntityConverter */
    protected $htmlEntityConverter;

    /**
     * ArticleTransformer constructor.
     * @param CategoryClient $categoryClient
     */
    public function __construct(CategoryClient $categoryClient)
    {
        $this->categoryClient = $categoryClient;
    }

    /**
     * @param string|\Move\I18n\I18nLanguageCode $language
     * @return \Cms\Client\Article\Transformer\ArticleTransformer
     * @throws \InvalidArgumentException
     */
    public function withLanguage($language) : ArticleTransformer
    {
        $language = (string)$language;
        if (!I18nLanguageCode::isValid($language)) {
            throw new \InvalidArgumentException('languagecode is invalide');
        }
        $clone = clone $this;
        $clone->language = $language;
        return $clone;
    }

    /**
     * @param ArticleRestfulModel|DomainObjectInterface $object
     * @return array
     */
    public function transform(DomainObjectInterface $object)
    {
        $result = parent::transform($object);
        if (!$object instanceof ArticleRestfulModel) {
            return $result;
        }

        // récuperation de la cagtegory
        if (!empty($object->category_id)) {
            try {
                $category = static::$categoryStore[$object->category_id]
                    ?? $this->categoryClient->fetchItem($object->category_id);
                static::$categoryStore[$object->category_id] = $category;
                if ($category instanceof CategoryModel) {
                    $result['category'] = (new ObjectTransformer())->transform($category);
                }
            } catch (HttpRequestException $e) {
                trigger_error(
                    'fetch category ' . $object->category_id . ' failed : ' . $e,
                    E_USER_WARNING
                );
            }
        }

        // modification des date
        $result['created_stamp'] = $object->created_at->getTimestamp();
        $result['modified_stamp'] = $object->modified_at->getTimestamp();

        $result['last_published_stamp'] = null;
        if ($object->last_published_at instanceof \DateTime) {
            $result['last_published_stamp'] = $object->last_published_at->getTimestamp();
        }

        $result['first_published_stamp'] = null;
        if ($object->first_published_at instanceof \DateTime) {
            $result['first_published_stamp'] = $object->first_published_at->getTimestamp();
        }

        // ajoute l'image de galerie
        $result['gallery_assets'] = [];
        $galleryAsset = $this->getAssetType($result['assets'], AssetCoverType::COVER_GALERIE);
        if (!empty($galleryAsset)) {
            $result['gallery_assets'] = $galleryAsset;
        }

        // ajout de l'image de presentation
        $result['main_asset'] = [];
        $mainAsset = $this->getAssetType($result['assets'], AssetCoverType::COVER_MAIN);
        if (!empty($mainAsset)) {
            $result['main_asset'] = $mainAsset[0];
        }

        // ajoute l'image de liste
        $result['list_asset'] = [];
        $listAsset = $this->getAssetType($result['assets'], AssetCoverType::COVER_LIST);
        if (!empty($listAsset)) {
            $result['list_asset'] = $listAsset[0];
        }

        $rawContent = $object->content;
        $rawCoverAbstract = $object->cover_abstract;
        $rawAbstract = $object->abstract;
        try {
            if ($this->language) {
                $rawAbstract = I18nHelper::translateObject($object, 'abstract', null, $this->language)
                    ?: $object->abstract;
                $rawCoverAbstract = I18nHelper::translateObject(
                    $object,
                    'cover_abstract',
                    null,
                    $this->language
                ) ?: $rawCoverAbstract;
                $rawContent = I18nHelper::translateObject($object, 'content', null, $this->language)
                    ?: $rawContent;
            }
        } catch (\Exception $e) {
            trigger_error('cannot translate article content : ' . $e->getMessage(), E_USER_WARNING);
        }

        // Construction du HTML du texte
        $result['html_content'] = $this->serializeContent($rawContent, '<p>&nbsp;</p>');

        // Construction du HTML du chapo de cover
        $result['html_cover_abstract'] = $this->serializeContent($rawCoverAbstract);

        // Construction du HTML du chapo
        $result['html_abstract'] = $this->serializeContent($rawAbstract);

        return $result;
    }

    /**
     * @param DashParser $parser
     * @return string
     */
    protected function serializeParser(DashParser $parser)
    {
        $htmlEntityConverter = $this->htmlEntityConverter;
        $htmlBlocConverter = $this->htmlBlockConverter;
        $htmlStyleConverter = $this->htmlStyleConverter;

        $serializer = new HtmlSerializer(
            $parser->getEntityCollection(),
            $htmlEntityConverter,
            $htmlBlocConverter,
            $htmlStyleConverter
        );
        return $serializer->serialize($parser->getBlocksCollection());
    }

    /**
     * @param HtmlStyleConverter $htmlStyleConverter
     * @return $this
     */
    public function setHtmlStyleConverter(HtmlStyleConverter $htmlStyleConverter)
    {
        $this->htmlStyleConverter = $htmlStyleConverter;
        return $this;
    }

    /**
     * @param HtmlBlockConverter $htmlBlockConverter
     * @return $this
     */
    public function setHtmlBlockConverter(HtmlBlockConverter $htmlBlockConverter)
    {
        $this->htmlBlockConverter = $htmlBlockConverter;
        return $this;
    }

    /**
     * @param HtmlEntityConverter $htmlEntityConverter
     * @return $this
     */
    public function setHtmlEntityConverter(HtmlEntityConverter $htmlEntityConverter)
    {
        $this->htmlEntityConverter = $htmlEntityConverter;
        return $this;
    }

    /**
     * @param array $assets
     * @param AssetCoverType|string $assetCoverType
     * @return array
     */
    protected function getAssetType($assets, $assetCoverType)
    {
        $assets = array_filter($assets, function ($asset) use ($assetCoverType) {
            return isset($asset['cover_type'])
                && (string)$asset['cover_type'] === (string)$assetCoverType;
        });
        $assets = array_values($assets);
        return $assets;
    }
}
