<?php

namespace Cms\Client\Command\Campsite;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Campsite\Model\CampsiteRestfulModel;
use Cms\Client\Command\Campsite\Command\FetchBulkCampsiteCommand;
use Cms\Client\Command\CommandExceptionHandleTrait;
use Cms\Client\Service\CampsiteService;
use Cms\Command\FetchWithScopeCommand;
use Move\Cache\CacheAwareInterface;
use Move\Command\Exception\BadParameterException;
use Move\Command\Exception\NotFoundException;
use Move\Http\Client\Exception\HttpRequestException;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class CampsiteCommandHandler
 * @package Cms\Client\Command\Campsite
 */
class CampsiteCommandHandler implements
    CacheAwareInterface,
    LoggerAwareInterface
{

    use CacheHelperTrait;
    use LoggerAwareTrait;
    use CommandExceptionHandleTrait;

    /**
     * @var CampsiteService
     */
    protected $service;

    /**
     * CampsiteCommandHandler constructor.
     * @param CampsiteService $service
     */
    public function __construct(CampsiteService $service)
    {
        $this->service = $service;
    }

    /**
     * @param FetchBulkCampsiteCommand $command
     * @return array|\Cms\Client\Campsite\Model\CampsiteRestfulModel[]
     * @throws \RuntimeException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     * @throws \Move\Command\Exception\NotFoundException
     */
    public function handleFetchBulkCampsite(FetchBulkCampsiteCommand $command) : array
    {
        $campsiteClient = $this->service->getCampsiteClient();
        try {
            $campsites = $campsiteClient->fetchBulkItems($command->getIds());
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('CamspiteCmdHd_FetchBulkCampsite', $e);
        }
        return $campsites ?? [];
    }

    /**
     * @param \Cms\Command\FetchWithScopeCommand $command
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\NotFoundException
     * @throws \RuntimeException
     */
    public function handleFetchWithScope(FetchWithScopeCommand $command) : CampsiteRestfulModel
    {
        $identifier = $command->getId();
        if (!$identifier) {
            throw new BadParameterException('identifier is empty');
        }
        $scopeId = $command->getScopeId();
        if (!is_numeric($identifier) && (!$identifier xor !$scopeId)) {
            throw new BadParameterException('scopeid must be define if identifier is not numeric');
        }

        $campsiteClient = $this->service->getCampsiteClient();
        try {
            if (is_numeric($identifier)) {
                // récuperation du camping par slug
                $campsiteModel = $campsiteClient->fetchItem(
                    $identifier
                );
            } else {
                // récuperation du camping par slug
                $campsiteModel = $campsiteClient->fetchItemBySlug(
                    $scopeId,
                    $identifier
                );
            }
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('CampsiteCommandHandler', $e);
            $campsiteModel = null;
        }

        if (!$campsiteModel instanceof CampsiteRestfulModel) {
            throw new NotFoundException(
                'campsite with identifier ' . $identifier . ' does not exist'
            );
        }
        return $campsiteModel;
    }
}
