<?php


namespace Cms\Client\Command\Member;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Command\CommandExceptionHandleTrait;
use Cms\Client\Command\Member\Command\FetchMemberCommand;
use Cms\Client\Command\Member\Command\SetNewsletterCommand;
use Cms\Client\Member\MemberClient;
use Cms\Client\Member\MemberRewardClient;
use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Model\Member\MemberAttributeName;
use Cms\Model\Member\MemberRewardModel;
use Move\Cache\CacheAwareInterface;
use Move\Command\Exception\BadParameterException;
use Move\Command\Exception\CommandException;
use Move\Http\Client\Exception\HttpRequestException;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class MemberCommandHandler
 * @package Cms\Client\Command\Member
 */
class MemberCommandHandler implements
    LoggerAwareInterface,
    CacheAwareInterface
{

    use LoggerAwareTrait;
    use CacheHelperTrait;
    use CommandExceptionHandleTrait;

    /** @var MemberClient */
    private $memberClient;

    /** @var \Cms\Client\Member\MemberRewardClient */
    private $memberRewardClient;

    /**
     * MemberCommandHandler constructor.
     * @param MemberClient $memberClient
     * @param \Cms\Client\Member\MemberRewardClient $memberRewardClient
     */
    public function __construct(MemberClient $memberClient, MemberRewardClient $memberRewardClient)
    {
        $this->memberClient = $memberClient;
        $this->memberRewardClient = $memberRewardClient;
    }

    /**
     * @param \Cms\Client\Command\Member\Command\SetNewsletterCommand $command
     * @return \Cms\Client\Member\Model\MemberRestfulModel
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\CommandException
     * @throws \RuntimeException
     */
    public function handleSetNewsletter(SetNewsletterCommand $command)
    {
        // insert dans la bdd du cms
        $memberClient = $this->memberClient;
        try {
            $memberModel = $memberClient->getItemByEmail($command->getEmail(), true);
        } catch (\Exception $e) {
            throw new BadParameterException('member is invalid', null, $e);
        }

        if (!$memberModel) {
            $memberModel = new MemberRestfulModel();
            $memberModel->email = $command->getEmail();
        }

        try {
            $attrs = [
                MemberAttributeName::INSCRIT_NEWS => $command->isInfo() || $command->isPartner(),
                MemberAttributeName::INSCRIT_NEWS_PARTNER => $command->isPartner(),
                MemberAttributeName::INSCRIT_NEWS_INFO => $command->isInfo(),
                MemberAttributeName::VALID_DBOPTIN => false,
            ];
            foreach ($attrs as $attr => $val) {
                if ($val !== null) {
                    $memberModel->setAttributes($command->getScopeId(), $attr, $val, true);
                }
            }
            $memberClient->saveItem($memberModel);
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('MemberCmdHd_SetNewsletter', $e);
        } catch (CommandException $e) {
            throw $e;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage(), [
                'exception' => $e,
            ]);
            throw new \RuntimeException('error while recording newsletter info', null, $e);
        }

        try {
            if ($command->getRewardSource() && $command->getRewardCampaign()) {
                $memberRewardModel = new MemberRewardModel();
                $memberRewardModel->source = $command->getRewardSource();
                $memberRewardModel->campaign = $command->getRewardCampaign();
                $memberRewardModel->member_id = $memberModel->id;
                $this->memberRewardClient->saveItem($memberRewardModel);
            }
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('MemberCmdHd_SetNewsletterReward', $e);
        } catch (CommandException $e) {
            throw $e;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage(), [
                'exception' => $e,
            ]);
            throw new \RuntimeException('error while recording reward info', null, $e);
        }

        return $memberModel;
    }

    /**
     * @param FetchMemberCommand $command
     * @return MemberRestfulModel|null
     * @throws \RuntimeException
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Psr\Cache\InvalidArgumentException
     */
    public function handleFetchMember(FetchMemberCommand $command)
    {
        $objectId = $command->getId();
        return $this->decorateWithCache(function () use ($objectId) {
            // récuperation du plan
            $client = $this->memberClient;
            try {
                if (filter_var($objectId, FILTER_VALIDATE_EMAIL)) {
                    $object = $client->getItemByEmail($objectId);
                } elseif (is_numeric($objectId)) {
                    $object = $client->fetchItem($objectId);
                } else {
                    throw new BadParameterException('objectId manquant');
                }
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('MemberCmdHd_FetchMember', $e);
            } catch (CommandException $e) {
                throw $e;
            } catch (\Exception $e) {
                $this->logger->error('an error occured while getting member data', [
                    'exception' => $e,
                ]);
            }
            return $object ?? null;
        }, 'member^' . $objectId, 120);
    }
}
