<?php

namespace Cms\LinkResource\Provider\MyCamping;

use Cms\Client\AbstractClient;
use Cms\LinkResource\LinkResourceType;
use Cms\LinkResource\Provider\AbstractProviderAdapter;
use Cms\LinkResource\Provider\DataAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareTrait;
use Move\Iterator\MapIterator;

/**
 * Class MyCampingProvider
 * @package Cms\LinkResource\Provider\MyCamping
 */
class MyCampingProvider extends AbstractProviderAdapter implements
    SearchAdapterAwareInterface,
    DataAdapterAwareInterface
{

    use SearchAdapterAwareTrait;

    /* Identifier */
    const IDENTIFIER = 'mycamping';

    /** @var array */
    protected $types = [LinkResourceType::MC_TOPIC];

    /** @var string */
    protected $identifier = self::IDENTIFIER;

    /** @var AbstractClient */
    private $topicClient;

    /**
     * MyCampingProvider constructor.
     * @param AbstractClient $topicClient
     */
    public function __construct(AbstractClient $topicClient = null)
    {
        $this->topicClient = $topicClient;
    }

    /**
     * @param AbstractClient $topicClient
     * @return MyCampingProvider
     */
    public function setTopicClient(AbstractClient $topicClient)
    {
        $this->topicClient = $topicClient;
        return $this;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param mixed $query
     * @return string[]
     * @throws \UnexpectedValueException
     */
    protected function execSearch($typeCode, $query)
    {
        // On récupère le bon typeCode
        switch ($typeCode) {
            case LinkResourceType::MC_TOPIC:
                if ($this->topicClient) {
                    $matches = $this->topicClient->getIndexWithParams([
                        'name' => $query,
                    ]);
                    $matches = new MapIterator($matches, function ($topic) {
                        return $topic['id'];
                    });
                    return iterator_to_array($matches, false);
                }

                // On checke l'url
                if (!filter_var($query, FILTER_VALIDATE_URL)) {
                    throw new \UnexpectedValueException('query must be an url');
                }

                // On checke que c'est bien une url lié à un topic
                $path = parse_url($query, PHP_URL_PATH);
                if (!preg_match('@^/[a-z]{2}/topic/(\d+)/@', $path, $matches)) {
                    throw new \UnexpectedValueException('query must be an url for topic');
                }

                // On envoie l'identifiant du topic après avoir checké qu'il existe
                if (empty($matches)) {
                    throw new \UnexpectedValueException('id topic not found');
                }
                return [$matches[1]];
                break;
            default:
                throw new \UnexpectedValueException('typeCode must be in the list: mc-topic');
                break;
        }
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param int $objectId
     * @return mixed
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getData($typeCode, $objectId)
    {
        switch ($typeCode) {
            case LinkResourceType::MC_TOPIC:
                if ($this->topicClient) {
                    $topicModel = $this->topicClient->fetchItem($objectId);
                    return $topicModel;
                }
                break;
        }
        return null;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param int $objectId
     * @return string
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getLabel($typeCode, $objectId)
    {
        switch ($typeCode) {
            case LinkResourceType::MC_TOPIC:
                if ($this->topicClient) {
                    $topicModel = $this->topicClient->fetchItem($objectId);
                    if ($topicModel) {
                        return $topicModel['name']['FR'];
                    }
                }
                return 'Topic id:' . $objectId;
                break;
        }
        return null;
    }
}
