<?php

namespace Move\Utils;

/**
 * Class String
 * @package Move\Utils
 */
class Str
{
    /**
     * @param int $length
     * @param bool|false $add_dashes
     * @param string $available_sets
     * @return string
     */
    public static function generatePasswd(
        $length = 9,
        $add_dashes = false,
        $available_sets = 'luds'
    ) : string {
        $sets = array();
        if (strpos($available_sets, 'l') !== false) {
            $sets[] = 'abcdefghjkmnpqrstuvwxyz';
        }
        if (strpos($available_sets, 'u') !== false) {
            $sets[] = 'ABCDEFGHJKMNPQRSTUVWXYZ';
        }
        if (strpos($available_sets, 'd') !== false) {
            $sets[] = '23456789';
        }
        if (strpos($available_sets, 's') !== false) {
            $sets[] = '!@#$%&*?';
        }
        $all = '';
        $password = '';
        foreach ($sets as $set) {
            $password .= $set[array_rand(str_split($set))];
            $all .= $set;
        }
        $all = str_split($all);
        for ($i = 0; $i < $length - count($sets); $i++) {
            $password .= $all[array_rand($all)];
        }
        $password = str_shuffle($password);
        if (!$add_dashes) {
            return $password;
        }
        $dash_len = floor(sqrt($length));
        $dash_str = '';
        while (\strlen($password) > $dash_len) {
            $dash_str .= substr($password, 0, $dash_len) . '-';
            $password = substr($password, $dash_len);
        }
        $dash_str .= $password;
        return $dash_str;
    }

    /**
     * @param string $str
     * @param int $limit
     * @param string|null $completion
     * @return string
     */
    public static function wordwrap(string $str, int $limit, string $completion = '...') : string
    {
        $len = mb_strlen($str);
        if ($len < $limit) {
            $completion = '';
        }
        $limit -= \strlen($completion);
        if (preg_match('/^.{1,' . $limit . '}\s/su', $str, $match)) {
            return $match[0] . $completion;
        }
        return mb_substr($str, 0, $limit) . $completion;
    }

    /**
     * @param string $str
     * @param int $limit
     * @param string|null $completion
     * @return string
     */
    public static function strwrap(string $str, int $limit, $completion = '...') : string
    {
        $len = mb_strlen($str);
        if ($len < $limit) {
            $completion = '';
        }
        $limit -= \strlen($completion);
        return mb_substr($str, 0, $limit) . $completion;
    }

    /**
     * @param string $string
     * @return string
     */
    public static function camelize($string) : string
    {
        return ucfirst(preg_replace_callback('/[-_]([a-z])/', function ($matches) {
            return strtoupper($matches[1]);
        }, $string));
    }

    /**
     * @param string $string
     * @return string
     */
    public static function underscore($string) : string
    {
        return preg_replace_callback('/([a-z])([A-Z])/', function ($matches) {
            return $matches[1] . '_' . strtolower($matches[2]);
        }, $string);
    }

    /**
     * string cleaner for url generator
     * @see http://cubiq.org/the-perfect-php-clean-url-generator
     * @param string $str
     * @param array|null $replace
     * @param string $delimiter
     * @return string
     */
    public static function toAscii($str, $replace = null, $delimiter = '-') : string
    {
        if (!empty($replace)) {
            $str = str_replace((array)$replace, ' ', $str);
        }

        $clean = iconv('UTF-8', 'ASCII//TRANSLIT', $str);
        $clean = preg_replace("/[^a-zA-Z0-9\/_|+ -]/", '', $clean);
        $clean = strtolower(trim($clean, '-'));
        $clean = preg_replace("/[\/_|+ -]+/", $delimiter, $clean);

        return trim($clean, $delimiter);
    }

    /**
     * @param mixed $data
     * @param bool $assoc
     * @param int $depth
     * @param int $decodeOptions
     * @return array|\stdClass
     * @throws \InvalidArgumentException
     */
    public static function parseJson($data, $assoc = true, $depth = 512, $decodeOptions = 0)
    {
        // Clear json_last_error()
        json_encode(null);

        $data = \json_decode((string)$data, $assoc, $depth, $decodeOptions);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \InvalidArgumentException(sprintf(
                'Unable to decode JSON to data in %s: %s',
                __CLASS__,
                json_last_error_msg()
            ));
        }

        return $data;
    }

    /**
     * @param mixed $data
     * @param int $encodeOptions
     * @throws \InvalidArgumentException
     * @return string
     */
    public static function toJson($data, $encodeOptions = 79) : string
    {
        if (\is_resource($data)) {
            throw new \InvalidArgumentException('Cannot JSON encode resources');
        }

        // Clear json_last_error()
        json_encode(null);

        $json = json_encode($data, $encodeOptions);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \InvalidArgumentException(sprintf(
                'Unable to encode data to JSON in %s: %s',
                __CLASS__,
                json_last_error_msg()
            ));
        }

        return $json;
    }
}
