<?php


namespace Cms\Client\Command\Geodata;

use Cms\Client\CmsApiIndexIterator;
use Cms\Client\Command\Article\FetchLinkedTypeCommandHandlerTrait;
use Cms\Client\Command\Geodata\Command\FetchAdminCommand;
use Cms\Client\Command\Geodata\Command\FetchAllAdminByFilterCommand;
use Cms\Client\Command\Geodata\Command\FetchCityCommand;
use Cms\Client\Command\Geodata\Command\FetchCountryCommand;
use Cms\Client\Command\Geodata\Command\FetchLinkedAdminCommand;
use Cms\Client\Geodata\AdminClient;
use Cms\Client\Geodata\CityClient;
use Cms\Client\Geodata\CountryClient;
use Cms\Client\Geodata\Model\AdminRestfulModel;
use Cms\Client\Geodata\Model\CityRestfulModel;
use Cms\Client\Service\ArticleService;
use Cms\Client\Service\ArticleServiceAwareTrait;
use Cms\LinkResource\LinkResourceHandler;
use Cms\Model\Geodata\AdminModel;
use Cms\Model\Geodata\CityModel;
use Cms\Model\Geodata\CountryModel;
use Move\Cache\CacheAwareInterface;
use Move\Command\Exception\BadParameterException;
use Move\Command\Exception\NotFoundException;
use Move\Http\Client\Exception\HttpRequestException;
use Psr\Cache\CacheItemInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class GeodataCommandHandler
 * @package Cms\Client\Command\Geodata
 */
class GeodataCommandHandler implements
    LoggerAwareInterface,
    CacheAwareInterface
{

    use FetchLinkedTypeCommandHandlerTrait;
    use ArticleServiceAwareTrait;
    use LoggerAwareTrait;

    /** @var \Cms\Client\Geodata\AdminClient */
    private $adminClient;

    /** @var \Cms\Client\Geodata\CityClient */
    private $cityClient;

    /** @var \Cms\Client\Geodata\CountryClient */
    private $countryClient;

    /** @var \Cms\LinkResource\LinkResourceHandler */
    private $resourceHandler;

    /**
     * GeodataCommandHandler constructor.
     * @param \Cms\Client\Geodata\AdminClient $adminClient
     * @param \Cms\Client\Geodata\CityClient $cityClient
     * @param \Cms\Client\Geodata\CountryClient $countryClient
     * @param \Cms\Client\Service\ArticleService $articleService
     * @param \Cms\LinkResource\LinkResourceHandler $resourceHandler
     */
    public function __construct(
        AdminClient $adminClient,
        CityClient $cityClient,
        CountryClient $countryClient,
        ArticleService $articleService,
        LinkResourceHandler $resourceHandler
    ) {
        $this->adminClient = $adminClient;
        $this->cityClient = $cityClient;
        $this->countryClient = $countryClient;
        $this->articleService = $articleService;
        $this->resourceHandler = $resourceHandler;
    }

    /**
     * @param \Cms\Client\Command\Geodata\Command\FetchLinkedAdminCommand $command
     * @return AdminRestfulModel[]
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \UnexpectedValueException
     * @throws \InvalidArgumentException
     */
    public function handleFetchLinkedAdmin(FetchLinkedAdminCommand $command)
    {
        $results = $this->handleFetchLinkedType($command);

        $sortArray = [];
        foreach ($results as $result) {
            try {
                $adminModel = $this->resourceHandler->getDataByModel($result);
                if ($adminModel instanceof AdminModel) {
                    $sortArray[$adminModel->slug] = $adminModel;
                }
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('GeodataCmdHd_FetchLinkedAdmin', $e);
            }
        }
        if (!$command->hasMostUsed()) {
            ksort($sortArray);
        }
        $results = array_values($sortArray);

        return $results;
    }

    /**
     * @param FetchAllAdminByFilterCommand $command
     * @return \Cms\Client\CmsApiIndexIterator|\Cms\Client\Geodata\Model\AdminRestfulModel[]
     * @throws \Move\Command\Exception\BadParameterException
     */
    public function handleFetchAllAdminByFilter(FetchAllAdminByFilterCommand $command)
    {
        $countryCode = $command->getCountryCode();
        if (empty($countryCode)) {
            throw new BadParameterException('country_code is empty');
        }

        $adminSlug = $command->getAdminSlug();
        $adminCode = $command->getAdminCode();
        if (empty($adminCode) && empty($adminSlug)) {
            throw new BadParameterException('admin_code & admin_slug is empty');
        }

        $adminLevel = $command->getAdminLevel();
        if (!$adminLevel) {
            throw new BadParameterException('admin_level is empty');
        }

        try {
            $adminModels = $this->adminClient->getBuilder()
                ->setFilter('country_code', $countryCode)
                ->setFilter('code', $adminCode)
                ->setFilter('slug', $adminSlug)
                ->setFilter('level', $adminLevel)
                ->getIndex(-1);
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('GeodataCmdHd_FetchAllAdmin', $e);
            $adminModels = CmsApiIndexIterator::createEmpty();
        }

        return $adminModels;
    }

    /**
     * @param FetchAdminCommand $command
     * @return AdminRestfulModel|null
     * @throws \Psr\Cache\InvalidArgumentException
     * @throws \RuntimeException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\NotFoundException
     */
    public function handleFetchAdmin(FetchAdminCommand $command)
    {
        $id = $command->getId();
        if (!$id) {
            throw new BadParameterException('id');
        }

        /** @var CacheItemInterface $cacheItem */
        if (true === $this->hitFromCache('geo^admin^' . $id, 3600 * 24, $cacheItem)) {
            $adminModel = $cacheItem->get();
        } else {
            try {
                $adminModel = $this->adminClient->fetchItem($id);
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('GeodataCmdHd_FetchAdmin', $e);
                $adminModel = null;
            }
            // prise en charge du cache
            $this->setInCache($cacheItem, $adminModel);
        }

        if (!$adminModel instanceof AdminRestfulModel) {
            throw new NotFoundException('admin:' . $id);
        }
        return $adminModel;
    }

    /**
     * @param FetchCityCommand $command
     * @return CityModel|null
     * @throws \Psr\Cache\InvalidArgumentException
     * @throws \RuntimeException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\NotFoundException
     */
    public function handleFetchCity(FetchCityCommand $command)
    {
        $id = $command->getId();
        $slugPart = $command->getSlugPart();
        if (!$id && !$slugPart) {
            throw new BadParameterException('id');
        }

        $admin1Code = $command->getAdmin1Code();
        $admin2Code = $command->getAdmin2Code();
        $cacheKey = 'geo^city^' . ($id ?: ($slugPart . '^' . $admin1Code . '^' . $admin2Code));
        /** @var CacheItemInterface $cacheItem */
        if (true === $this->hitFromCache($cacheKey, 3600 * 24, $cacheItem)) {
            $cityModel = $cacheItem->get();
        } else {
            try {
                $adminClient = $this->cityClient;
                if ($id) {
                    $cityModel = $adminClient->fetchItem($id);
                } else {
                    $results = $adminClient->getBuilder()
                        ->setFilter('slug', $slugPart)
                        ->setFilter('admin1_code', $admin1Code)
                        ->setFilter('admin2_code', $admin2Code)
                        ->getIndex(0);
                    $cityModel = null;
                    if ($results->getTotal() === 1) {
                        $cityModel = $results->getFirst();
                    }
                }
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('GeodataCmdHd_FetchCity', $e);
                $cityModel = null;
            }
            // prise en charge du cache
            $this->setInCache($cacheItem, $cityModel);
        }

        if (!$cityModel instanceof CityRestfulModel) {
            throw new NotFoundException('city:' . ($id ?: $slugPart) . ' (' . ($admin1Code . '/' . $admin2Code) . ')');
        }
        return $cityModel;
    }

    /**
     * @param FetchCountryCommand $command
     * @return CountryModel|null
     * @throws \Psr\Cache\InvalidArgumentException
     * @throws \RuntimeException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\NotFoundException
     */
    public function handleFetchCountry(FetchCountryCommand $command)
    {
        $id = $command->getId();
        if (!$id) {
            throw new BadParameterException('id');
        }

        /** @var CacheItemInterface $cacheItem */
        if (true === $this->hitFromCache('geo^country^' . $id, 3600 * 24, $cacheItem)) {
            $countryModel = $cacheItem->get();
        } else {
            try {
                $countryClient = $this->countryClient;
                $countryModel = $countryClient->fetchItem($id);
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('GeodataCmdHd_FetchCountry', $e);
                $countryModel = null;
            }
            // prise en charge du cache
            $this->setInCache($cacheItem, $countryModel);
        }

        if (!$countryModel instanceof CountryModel) {
            throw new NotFoundException('country:' . $id);
        }
        return $countryModel;
    }
}
