<?php


namespace Cms\Controller;

use Cms\Client\Asset\Model\AssetResfulModel;
use Cms\Client\Command\CommandExceptionHandleTrait;
use Cms\Client\Service\AssetService;
use Cms\Model\Asset\AssetProfil;
use Cms\Model\Asset\Config\ConfigFactory;
use Cms\Model\Asset\Config\ConfigImageModel;
use Cms\Model\Asset\Config\ConfigModel;
use Cms\Utils\AssetUtils;
use League\Flysystem\Filesystem;
use Move\Http\Client\Exception\HttpRequestException;
use Move\Http\RequestUtils;
use Move\Http\ResponseUtils;
use Move\Image\ImageUtils;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Zend\Diactoros\Stream;

/**
 * Class AbstractProxyAssetController
 * @package Cms\Controller
 */
abstract class AbstractProxyAssetController implements LoggerAwareInterface
{

    use LoggerAwareTrait;
    use CommandExceptionHandleTrait;

    /**
     * @var AssetService
     */
    private $assetService;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * UtilsController constructor.
     * @param AssetService $assetService
     * @param Filesystem $filesystem
     */
    public function __construct(
        AssetService $assetService,
        Filesystem $filesystem
    ) {
        $this->assetService = $assetService;
        $this->filesystem = $filesystem;
        $this->logger = new NullLogger();
    }

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param $vars
     * @return mixed
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public function proxyImage(ServerRequestInterface $request, ResponseInterface $response, $vars)
    {
        if (empty($vars['asset_id'])) {
            return $this->getPlaceholder($response, 'no filename', 800, 600);
        }

        // lecture de l'api
        try {
            $assetModel = $this->assetService->getAssetClient()->fetchItem($vars['asset_id']);
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('ProxyAsset_AssetFetch', $e);
            return $this->getPlaceholder($response, 'api failed', 800, 600);
        }

        if (!$assetModel instanceof AssetResfulModel) {
            return $this->getPlaceholder($response, 'asset invalid', 800, 600);
        }

        // check si le fichier existe sur le disque
        $filepath = $assetModel->file_path;
        if (!$this->filesystem->has($filepath)) {
            return $this->getPlaceholder(
                $response,
                'not exist on filesystem : ' . $filepath,
                800,
                600
            );
        }

        // récuperation de la resource
        $file = $this->filesystem->get($filepath);
        $content = $file->read();
        $lastMod = $file->getTimestamp();
        $mimeType = $file->getMimetype();

        // si pas une image on renvoi le contnenu direct
        if ((string)$assetModel->profil !== AssetProfil::IMAGE) {
            $contentStream = $this->rawToStream($content);
        } // cas des fichier gif animé
        elseif (ImageUtils::isAnimated($content)) {
            $contentStream = $this->rawToStream($content);
        } // pour les autres fichier images
        else {
            //error_log("memory before create image : " . memory_get_usage()/1024);
            $img = ImageUtils::createImage($content, $mimeType);
            //error_log("memory after create image : " . memory_get_usage()/1024);

            // lecture de la config
            if (isset($vars['config_id']) && !empty($vars['config_id'])) {
                $config = $this->getConfigFromAssetModel($assetModel, $vars['config_id']);
                if ($config === null) {
                    return $this->getPlaceholder(
                        $response,
                        'config id does not exist : ' . $vars['config_id'],
                        800,
                        600
                    );
                }

                // lecture de la config d'image
                $imgCopy = $img;
                if ($config->config instanceof ConfigImageModel) {
                    $imgCopy = AssetUtils::scaleImageWithConfig($img, $config->config, $vars);
                    //error_log("memory after scale image config : " . memory_get_usage()/1024);
                }
            } // check du nocrop
            elseif (RequestUtils::getQueryValue($request, 'nocrop') === '1') {
                // recuperation des infos de l'image
                $imgWidth = imagesx($img);
                $imgHeight = imagesy($img);
                $ratio = $imgHeight / $imgWidth;
                // si une largeur
                if (!empty($vars['width'])) {
                    $pageWidth = $vars['width'];
                    // check des palier de largeur
                    $pageHeight = $pageWidth * $ratio;
                }
                if (!empty($vars['height'])) {
                    if (empty($pageHeight) || $pageHeight > $vars['height']) {
                        $pageHeight = $vars['height'];
                    }
                    $pageWidth = $pageHeight / $ratio;
                }

                // scaling
                if (!empty($pageWidth) && !empty($pageHeight)) {
                    $imgCopy = ImageUtils::scaleImage($img, $pageWidth, $pageHeight);
                } else {
                    $imgCopy = ImageUtils::scaleImageCut($img, $vars);
                }
                //error_log("memory after scale image : " . memory_get_usage()/1024);
                if ($imgCopy === null) {
                    return $this->getPlaceholder($response, 'resize failed', 800, 600);
                }
            } else {
                // scaling
                $imgCopy = ImageUtils::scaleImageCut($img, $vars);
                //error_log("memory after scaleCut image : " . memory_get_usage()/1024);
                if ($imgCopy === null) {
                    return $this->getPlaceholder($response, 'resize failed', 800, 600);
                }
            }

            // conversion de l'image en resource stream
            $contentStream = $this->imageToStream($imgCopy, $mimeType);
        }

        // lecture du fichier sur le disque
        $response = $response
            ->withBody($contentStream)
            ->withHeader('Content-Type', $mimeType);

        // ajoute le ttl
        $response = ResponseUtils::withLastModified($response, $lastMod, 24 * 3600 * 60);
        return $response;
    }

    /**
     * @param ResponseInterface $response
     * @param string $message
     * @param int $pageWidth
     * @param int $pageHeight
     * @return ResponseInterface
     */
    abstract protected function getPlaceholder(
        ResponseInterface $response,
        $message,
        $pageWidth,
        $pageHeight
    );

    /**
     * @param string $content
     * @return Stream
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    protected function rawToStream($content) : Stream
    {
        // conversion de l'image en resource stream
        $stream = new Stream('php://memory', 'w+');
        $stream->write($content);
        $stream->rewind();
        return $stream;
    }

    /**
     * @param resource $img
     * @param string $mimeType
     * @return Stream
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    protected function imageToStream($img, $mimeType) : Stream
    {
        // conversion de l'image en resource stream
        $content = ImageUtils::imageToString($img, $mimeType);
        return $this->rawToStream($content);
    }

    /**
     * @param AssetResfulModel $assetModel
     * @param int $configId
     * @return ConfigModel|null
     * @throws \InvalidArgumentException
     */
    protected function getConfigFromAssetModel(AssetResfulModel $assetModel, $configId)
    {
        foreach ($assetModel->configs as $assetConfig) {
            if ((int)$assetConfig['id'] === (int)$configId) {
                return ConfigFactory::createConfigFromAsset($assetModel, $assetConfig);
                break;
            }
        }
        return null;
    }
}
