<?php

namespace Cms;

use Monolog\Handler\ChromePHPHandler;
use Monolog\Logger;
use Move\Environment;
use Move\Log\LogstashHandlerFactory;
use Move\Log\PapertailHandlerFactory;
use Move\Log\RollbarHandlerFactory;
use Move\Providers\LoggerProvider;

/**
 * Class LoggerFactory
 * @package Cms
 */
class LoggerFactory
{

    /**
     * @param \Move\Environment $env
     * @return array
     */
    private static function getLogLevelMap(Environment $env) : array
    {
        // definition par default
        $browser = Logger::WARNING;
        $system = Logger::ERROR;
        $external = null;

        // definition du log system en info si sur debug env
        if ($env->isDebug()) {
            $system = Logger::INFO;
        }

        // surcharge le log level pour toute l'app si defini
        $appLogLevel = \defined('APP_LOG_LEVEL') ? APP_LOG_LEVEL : null;
        if ($appLogLevel !== null) {
            $browser = $system = $appLogLevel;
        }

        // surcharge pour le browser si defini
        $browser = \defined('BROWSER_LOG_LEVEL') ? BROWSER_LOG_LEVEL : $browser;

        // surcharge pour le system si defini
        $system = \defined('SYSTEM_LOG_LEVEL') ? SYSTEM_LOG_LEVEL : $system;

        // surcharge pour le system si defini
        $external = \defined('EXTERNAL_LOG_LEVEL') ? EXTERNAL_LOG_LEVEL : $external;

        return ['system' => $system, 'external' => $external, 'browser' => $browser];
    }

    /**
     * @param \Move\Environment $env
     * @param string $loggerName
     * @param string $rollbarToken
     * @param string $rootPath
     * @param string|null $logstashHost
     * @param string|null $logstashPort
     * @param array $logLevelMap ['system' => int, 'external' => int, 'browser' => int]
     * @return \Move\Providers\LoggerProvider
     */
    public static function register(
        Environment $env,
        string $loggerName,
        string $rollbarToken = null,
        string $rootPath = null,
        string $logstashHost = null,
        string $logstashPort = null,
        array $logLevelMap = []
    ) : LoggerProvider {
        $handlers = [];

        // detect app log level
        $logLevel = array_merge(static::getLogLevelMap($env), $logLevelMap);

        // ajoute rollbar en preprod/prod
        if ($rollbarToken && $rootPath && ($env->isProd() || $env->isStaging())) {
            $rollbarHandler = RollbarHandlerFactory::create(
                $rollbarToken,
                $rootPath,
                $env,
                $logLevel['external']
            );
            array_unshift($handlers, $rollbarHandler);
        }

        // ajoute logstash en prod
        if ($logstashHost && $env->isProd()) {
            $logstashHandler = LogstashHandlerFactory::createGelfUdp(
                $logstashHost,
                $logstashPort
            );
            array_unshift($handlers, $logstashHandler);
        }

        return new LoggerProvider($loggerName, $handlers, $logLevel['system']);
    }
}
