<?php

namespace Cms\Route;

use Move\Utils\Str;

/**
 * Class AssetProxyRoute
 * @package Cms\Route
 */
class AssetProxyRoute extends AbstractRoute
{
    /** @var array */
    protected $routes;

    /** @var string */
    private $controllerClass;

    /** @var string */
    private $action;

    /**
     * AssetProxyRoute constructor.
     * @param $routePrefix
     * @param string $controllerClass
     * @param string $action
     */
    public function __construct($routePrefix, $controllerClass, $action = 'proxyImage')
    {
        $this->routes = [
            'full_config' => $routePrefix . self::getStdPath(true, true, true),
            'height_config' => $routePrefix . self::getStdPath(false, true, true),
            'width_config' => $routePrefix . self::getStdPath(true, false, true),
            'default_config' => $routePrefix . self::getStdPath(false, false, true),

            'full' => $routePrefix . self::getStdPath(true, true),
            'height' => $routePrefix . self::getStdPath(false, true),
            'width' => $routePrefix . self::getStdPath(true),
            'default' => $routePrefix . self::getStdPath(),

            'ph' => $routePrefix . self::getPhPath(),
        ];
        $this->controllerClass = $controllerClass;
        $this->action = $action;
    }

    /**
     * @return void
     */
    public function register()
    {
        // ajoutes les routes
        $this->addPageRoute('full_config', $this->controllerClass, 'get', $this->action);
        $this->addPageRoute('height_config', $this->controllerClass, 'get', $this->action);
        $this->addPageRoute('width_config', $this->controllerClass, 'get', $this->action);
        $this->addPageRoute('default_config', $this->controllerClass, 'get', $this->action);

        $this->addPageRoute('full', $this->controllerClass, 'get', $this->action);
        $this->addPageRoute('height', $this->controllerClass, 'get', $this->action);
        $this->addPageRoute('width', $this->controllerClass, 'get', $this->action);
        $this->addPageRoute('default', $this->controllerClass, 'get', $this->action);

        $this->addPageRoute('ph', $this->controllerClass, 'get', $this->action);

        // ajout du decorateur d'url
        $this->setBuilderDecorator([$this, 'decorate']);
    }

    /**
     * @param $identifier
     * @param \Cms\Route\PathBuilder $builder
     * @return \Cms\Route\PathBuilder
     * @throws \InvalidArgumentException
     */
    protected function decorate($identifier, PathBuilder $builder) : PathBuilder
    {
        $userData = $builder->getArrayCopy();

        // si les donnée mini ne sont pas passé on donne le ph
        if (empty($userData['asset_id'])) {
            $identifier = 'ph';
        } else {
            // detection des donnée passé
            $hasWidth = $hasHeight = false;
            if (!empty($userData['width']) && is_numeric($userData['width'])) {
                $hasWidth = true;
            }
            if (!empty($userData['height']) && is_numeric($userData['height'])) {
                $hasHeight = true;
            }

            // detection de l'identifier
            if ($hasWidth && $hasHeight) {
                $identifier = 'full';
            } else {
                if ($hasWidth) {
                    $identifier = 'width';
                } elseif ($hasHeight) {
                    $identifier = 'height';
                } else {
                    $identifier = $identifier ?: 'default';
                }
            }

            // detection si titre passé
            if (!empty($userData['asset_filename'])) {
                $fileExt = pathinfo($userData['asset_filename'], PATHINFO_EXTENSION);
                $fileName = pathinfo($userData['asset_filename'], PATHINFO_FILENAME);
                $fileName = Str::toAscii($fileName) . '.' . strtolower($fileExt);
                $userData['filename'] = $fileName;
            }

            // detection config
            if (!empty($userData['asset_config_id'])) {
                $userData['config_id'] = $userData['asset_config_id'];
                $identifier .= '_config';
            }
        }
        return $this
            ->getPathBuilder($identifier)
            ->mergeParams($userData);
    }

    /**
     * @param bool $useWidth
     * @param bool $useHeight
     * @param bool $useConfig
     * @return string
     */
    public static function getStdPath(
        $useWidth = false,
        $useHeight = false,
        $useConfig = false
    ) : string {
        $filenamePath = '[/{filename:.*}]';
        $configPath = '/{asset_id:number}' . ($useConfig ? '/config/{config_id:number}' : '');

        if ($useWidth && $useHeight) {
            return '/{width:number}x{height:number}' . $configPath . $filenamePath;
        }
        if ($useWidth) {
            return '/{width:number}' . $configPath . $filenamePath;
        }
        if ($useHeight) {
            return '/x{height:number}' . $configPath . $filenamePath;
        }
        return $configPath . $filenamePath;
    }

    /**
     * @return string
     */
    public static function getPhPath() : string
    {
        return '/ph';
    }
}
