<?php

namespace Move\Newsletter\Provider;

use Mailjet\Client;
use Mailjet\Resources;
use Mailjet\Response;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Class MailjetAdapter
 * @package Move\Newsletter\Provider
 */
class MailjetAdapter implements ProviderInterface, LoggerAwareInterface
{

    use MailjetClientServiceTrait;
    use LoggerAwareTrait;

    /** @var \Mailjet\Client */
    protected $client;

    /** @var string */
    protected $idList;

    /** @var array */
    protected $properties = [
        'abo' => '',
        'actifs' => '',
        'facebook' => '',
        'firstname' => 'Prenom',
        'gdpr' => '',
        'genre' => 'Civilite',
        'lastname' => 'Nom',
        'optin' => 'OptIn',
        'pdf' => 'OptInDocument',
        'signin_source' => '',
        'signin_campaign' => '',
    ];

    /**
     * MailjetAdapter constructor.
     * @param string $apiKey
     * @param string $apiSecret
     * @param string $idList
     * @param array $customProps
     */
    public function __construct(string $apiKey, string $apiSecret, string $idList, array $customProps)
    {
        $this->client = new Client($apiKey, $apiSecret, true);
        $this->idList = $idList;
        $this->logger = new NullLogger();
        $this->properties = array_merge($this->properties, array_fill_keys($customProps, ''));
    }

    /**
     * @param string $email
     * @param array $dataset
     * @return bool
     */
    public function update(string $email, array $dataset = []) : bool
    {
        $contactName = $dataset['name'] ?? null;
        unset($dataset['name']);
        // envoi du client dans la base mailjet
        $response = $this->client->post(Resources::$ContactslistManagecontact, [
            'body' => [
                'Action' => 'addforce',
                'Email' => $email,
                'Name' => $contactName,
                'Properties' => $this->normalizeDatas($dataset),
            ],
            'id' => $this->idList,
        ]);
        return $this->handleResponse($response, $this->logger);
    }

    /**
     * @param string $email
     * @param array $dataset
     * @return bool
     */
    public function insert(string $email, array $dataset = []) : bool
    {
        $contactName = $dataset['name'] ?? null;
        unset($dataset['name']);
        // envoi du client dans la base mailjet
        $response = $this->client->post(Resources::$ContactslistManagecontact, [
            'body' => [
                'Action' => 'addnoforce',
                'Email' => $email,
                'Name' => $contactName,
                'Properties' => $this->normalizeDatas($dataset),
            ],
            'id' => $this->idList,
        ]);
        return $this->handleResponse($response, $this->logger);
    }

    /**
     * @param string $email
     * @return bool
     */
    public function delete(string $email) : bool
    {
        // envoi du client dans la base mailjet
        $response = $this->client->post(Resources::$ContactslistManagecontact, [
            'body' => [
                'Action' => 'unsub',
                'Email' => $email,
            ],
            'id' => $this->idList,
        ]);
        return $this->handleResponse($response, $this->logger);
    }

    /**
     * @param string $email
     * @return bool
     */
    public function blacklist(string $email) : bool
    {
        $response = $this->client->put(Resources::$Contact, [
            'body' => [
                'IsExcludedFromCampaigns' => true,
                'id' => $email,
            ],
        ]);
        return $this->handleResponse($response, $this->logger);
    }

    /**
     * @param string $email
     * @return bool
     */
    public function unblacklist(string $email) : bool
    {
        $response = $this->client->put(Resources::$Contact, [
            'body' => [
                'IsExcludedFromCampaigns' => false,
                'id' => $email,
            ],
        ]);
        return $this->handleResponse($response, $this->logger);
    }

    /**
     * @param array $dataset
     * @return array
     */
    protected function normalizeDatas(array $dataset) : array
    {
        $normDataset = [];
        foreach ($dataset as $key => $data) {
            if (isset($this->properties[$key])) {
                $normDataset[$key] = $data;
            } elseif (false !== ($keyFound = \array_search($key, $this->properties, true))) {
                $normDataset[$keyFound] = $data;
            }
        }
        return $normDataset;
    }
}
