<?php

namespace Move\Template\Middleware;

/**
 * Class HtmlMinifier
 * @package Lightspeed\Transform
 */
class HtmlMinifier implements MiddlewareInterface
{

    /**
     * @var boolean
     */
    protected $jsCleanComments = true;

    /**
     * @var bool|null
     */
    protected $isXhtml;

    /**
     * @var null|string
     */
    protected $replacementHash;

    /**
     * @var array
     */
    protected $placeholders = [];

    /**
     * @var callable
     */
    protected $cssMinifier;

    /**
     * @var callable
     */
    protected $jsMinifier;


    /**
     * Create a minifier object
     * @param array $options
     * 'cssMinifier' : (optional) callback function to process content of STYLE
     * elements.
     * 'jsMinifier' : (optional) callback function to process content of SCRIPT
     * elements. Note: the type attribute is ignored.
     * 'jsCleanComments' : (optional) whether to remove HTML comments beginning and end of script
     *     block
     * 'xhtml' : (optional boolean) should content be treated as XHTML1.0? If
     */
    public function __construct(array $options = [])
    {
        if (isset($options['xhtml'])) {
            $this->isXhtml = (bool)$options['xhtml'];
        }
        if (isset($options['cssMinifier']) && \is_callable($options['cssMinifier'])) {
            $this->cssMinifier = $options['cssMinifier'];
        }
        if (isset($options['jsMinifier']) && \is_callable($options['jsMinifier'])) {
            $this->jsMinifier = $options['jsMinifier'];
        }
        if (isset($options['jsCleanComments'])) {
            $this->jsCleanComments = (bool)$options['jsCleanComments'];
        }
    }

    /**
     * @param string $template
     * @return string
     */
    public function __invoke($template)
    {
        return $this->process($template);
    }

    /**
     * Minify the markeup given in the constructor
     * @param string $content
     * @return string
     */
    public function process($content) : string
    {
        // detection html doctype
        $content = str_replace("\r\n", "\n", trim($content));
        if ($this->isXhtml === null) {
            $this->isXhtml = (false !== strpos(
                $content,
                '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML'
            ));
        }
        $this->replacementHash = 'MINIFYHTML' . md5($_SERVER['REQUEST_TIME']);
        $this->placeholders = [];

        // replace SCRIPTs (and minify) with placeholders
        $content = preg_replace_callback(
            '/(\\s*)<script(\\b[^>]*?>)([\\s\\S]*?)<\\/script>(\\s*)/i',
            [$this, 'removeScriptCB'],
            $content
        );

        // replace STYLEs (and minify) with placeholders
        $content = preg_replace_callback(
            '/\\s*<style(\\b[^>]*>)([\\s\\S]*?)<\\/style>\\s*/i',
            [$this, 'removeStyleCB'],
            $content
        );

        // remove HTML comments (not containing IE conditional comments or #).
        $content = preg_replace_callback(
            '/<!--([\\s\\S]*?)-->/',
            [$this, 'commentCB'],
            $content
        );

        // replace PREs with placeholders
        $content = preg_replace_callback('/\\s*<pre(\\b[^>]*?>[\\s\\S]*?<\\/pre>)\\s*/i', [
            $this,
            'removePreCB',
        ], $content);

        // replace TEXTAREAs with placeholders
        $content = preg_replace_callback(
            '/\\s*<textarea(\\b[^>]*?>[\\s\\S]*?<\\/textarea>)\\s*/i',
            [$this, 'removeTextareaCB'],
            $content
        );

        // No space
        $search = [
            '/\>[^\S ]+/s',  // strip whitespaces after tags, except space
            '/[^\S ]+\</s',  // strip whitespaces before tags, except space
            '/\s+/s'       // shorten multiple whitespace sequences
        ];
        $content = preg_replace($search, ['>', '<', ' '], $content);

        // remove ws around block/undisplayed elements
        $content = preg_replace('/\\s+(<\\/?(?:area|article|aside|base(?:font)?|blockquote|body'
            . '|canvas|caption|center|col(?:group)?|dd|dir|div|dl|dt|fieldset|figcaption|figure|footer|form'
            . '|frame(?:set)?|h[1-6]|head|header|hgroup|hr|html|legend|li|link|main|map|menu|meta|nav'
            . '|ol|opt(?:group|ion)|output|p|param|section|t(?:able|body|head|d|h||r|foot|itle)'
            . '|ul|video)\\b[^>]*>)/i', '$1', $content);

        // remove ws outside of all elements
        $content = preg_replace(
            '/>(\\s(?:\\s*))?([^<]+)(\\s(?:\s*))?</',
            '>$1$2$3<',
            $content
        );

        // fill placeholders
        $content = strtr($content, $this->placeholders);
        // issue 229: multi-pass to catch scripts that didn't get replaced in textareas
        $content = strtr($content, $this->placeholders);
        return $content;
    }

    /**
     * @param array $m
     * @return string
     */
    protected function commentCB($m) : string
    {
        return (0 === strpos($m[1], '[') || false !== strpos($m[1], '<!['))
            ? $m[0]
            : '';
    }

    /**
     * @param string $content
     * @return string
     */
    protected function reservePlace($content) : string
    {
        $placeholder = '%' . $this->replacementHash . count($this->placeholders) . '%';
        $this->placeholders[$placeholder] = $content;
        return $placeholder;
    }

    /**
     * @param string $m
     * @return string
     */
    protected function removePreCB($m) : string
    {
        return $this->reservePlace("<pre{$m[1]}");
    }

    /**
     * @param string $m
     * @return string
     */
    protected function removeTextareaCB($m) : string
    {
        return $this->reservePlace("<textarea{$m[1]}");
    }

    /**
     * @param string $m
     * @return string
     */
    protected function removeStyleCB($m) : string
    {
        $openStyle = "<style{$m[1]}";
        $css = $m[2];
        // remove HTML comments
        $css = preg_replace('/(?:^\\s*<!--|-->\\s*$)/', '', $css);
        // remove CDATA section markers
        $css = $this->removeCdata($css);
        // minify
        $css = \call_user_func($this->cssMinifier ?: 'trim', $css);
        return $this->reservePlace($this->needsCdata($css)
            ? "{$openStyle}/*<![CDATA[*/{$css}/*]]>*/</style>"
            : "{$openStyle}{$css}</style>");
    }

    /**
     * @param string $m
     * @return string
     */
    protected function removeScriptCB($m) : string
    {
        $openScript = "<script{$m[2]}";
        $js = $m[3];
        // whitespace surrounding? preserve at least one space
        $ws1 = ($m[1] === '') ? '' : ' ';
        $ws2 = ($m[4] === '') ? '' : ' ';
        // remove HTML comments (and ending "//" if present)
        if ($this->jsCleanComments) {
            $js = preg_replace('/(?:^\\s*<!--\\s*|\\s*(?:\\/\\/)?\\s*-->\\s*$)/', '', $js);
        }
        // remove CDATA section markers
        $js = $this->removeCdata($js);
        // minify
        $js = \call_user_func($this->jsMinifier ?: 'trim', $js);
        return $this->reservePlace($this->needsCdata($js)
            ? "{$ws1}{$openScript}/*<![CDATA[*/{$js}/*]]>*/</script>{$ws2}"
            : "{$ws1}{$openScript}{$js}</script>{$ws2}");
    }

    /**
     * @param string $str
     * @return string
     */
    protected function removeCdata($str) : string
    {
        return (false !== strpos($str, '<![CDATA['))
            ? str_replace(array('<![CDATA[', ']]>'), '', $str)
            : $str;
    }

    /**
     * @param string $str
     * @return bool
     */
    protected function needsCdata($str) : bool
    {
        return ($this->isXhtml && preg_match('/(?:[<&]|\\-\\-|\\]\\]>)/', $str));
    }
}
