<?php

namespace Cms\Client\Version;

use Cms\Client\AbstractClient;
use Cms\LinkResource\LinkResourceType;
use Cms\LinkResource\Provider\CMS\CmsProvider;
use Cms\Model\Version\VersionModel;
use Cms\Model\Version\VersionState;
use Move\Utils\Str;

/**
 * Class VersionClient
 * @package Cms\Client\Version
 */
class VersionClient extends AbstractClient
{
    /** @var string */
    protected $basePath = 'version';

    /**
     * @param array $dataset
     * @return VersionModel
     */
    public function handleData($dataset = [])
    {
        return (new VersionModel())->populate($dataset, true);
    }

    /**
     * @param array $queryParams
     * @return \Cms\Client\CmsApiIndexIterator|VersionModel[]
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getIndexWithParams($queryParams)
    {
        $result = $this->loadIndexFromClient($this->basePath, $queryParams);
        return $result;
    }

    /**
     * @param int $userId
     * @param int $scopeId
     * @param array $dataset
     * @param string $refType
     * @param string $ref
     * @param string|null $versionState
     * @param string|null $provider
     * @return bool|null|VersionModel
     * @throws \InvalidArgumentException
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \RuntimeException
     * @throws \UnexpectedValueException
     */
    public function saveVersion(
        int $userId,
        int $scopeId,
        array $dataset,
        $refType,
        $ref,
        $versionState = null,
        string $provider = null
    ) {
        // sauvegarde de la version du contrat
        $versionModel = (new VersionModel())->populate([
            'cms_user_id' => $userId,
            'scope_id' => $scopeId,
            'state' => (string)$versionState ?: VersionState::WAIT_MODERATE,
            'raw' => Str::toJson($dataset),
        ], true);
        $versionModel->link_provider = $provider ?: CmsProvider::IDENTIFIER;
        $versionModel->link_type = $refType;
        $versionModel->link_ref = $ref;
        return $this->saveItem($versionModel);
    }

    /**
     * @param int $scopeId
     * @param string|\Cms\LinkResource\LinkResourceType|null $resourceType
     * @param string|\Cms\Model\Version\VersionState|null $state
     * @param int $pageNum
     * @return \Cms\Client\CmsApiIndexIterator|VersionModel[]
     */
    public function fetchAllByScope($scopeId, $resourceType = null, $state = null, $pageNum = 0)
    {
        $filter = $this->getBuilder()
            ->setFilter('scope_id', $scopeId)
            ->setFilter('link_type', (string)$resourceType)
            ->setFilter('state', (string)$state);
        return $filter->getIndex($pageNum);
    }

    /**
     * @param int $scopeId
     * @param string|\Cms\LinkResource\LinkResourceType|null $linkType
     * @param string|\Cms\Model\CmsUser\CmsUserProfil[]|null $userProfil
     * @param string|null $linkRef
     * @return \Cms\Client\CmsApiIndexIterator|VersionModel[]
     */
    public function fetchAllForLog($scopeId, $linkType, $userProfil = null, $linkRef = null)
    {
        $filter = $this->getBuilder()
            ->setFilter('export_mode', 'log')
            ->setFilter('scope_id', $scopeId)
            ->setFilter('link_type', (string)$linkType)
            ->setFilter('link_ref', (string)$linkRef)
            ->setFilter('cms_user_profil', $userProfil);
        return $filter->getIndex(-1);
    }

    /**
     * @param int $cmsUserId
     * @param int $scopeId
     * @param null|string $linkType
     * @param VersionState|string|array|null $state
     * @param int $pageNum
     * @return \Cms\Client\CmsApiIndexIterator|\Cms\Model\Version\VersionModel[]
     */
    public function fetchAllByCmsUser(
        $cmsUserId,
        $scopeId,
        $linkType = null,
        $state = null,
        $pageNum = 0
    ) {
        $filter = $this->getBuilder()
            ->setFilter('scope_id', $scopeId)
            ->setFilter('cms_user_id', $cmsUserId);
        if ($linkType) {
            $filter->setFilter('link_type', $linkType);
        }
        if ($state) {
            $filter->setFilter('state', $state);
        }
        return $filter->getIndex($pageNum);
    }

    /**
     * @param string|LinkResourceType $type
     * @param string $ref
     * @param int|null $scopeId
     * @param VersionState|string|array|null $state
     * @param int $pageNum
     * @return \Cms\Client\CmsApiIndexIterator|VersionModel[]
     */
    public function fetchAllByResource($type, $ref, $scopeId = null, $state = null, $pageNum = 0)
    {
        $filter = $this->getBuilder()
            ->setFilter('scope_id', $scopeId)
            ->setFilter('link_type', $type)
            ->setFilter('link_ref', $ref);
        if ($state) {
            $filter->setFilter('state', $state);
        }
        return $filter->getIndex($pageNum);
    }
}
