<?php


namespace Cms\Search\Article;

use Cms\Search\ElasticQueryBuilderInterface;
use Cms\Search\ElasticQueryBuilderTrait;

/**
 * Class ArticleSearchQueryBuilder
 * @package Cms\Search\Article
 */
class ArticleSearchQueryBuilder implements ElasticQueryBuilderInterface
{
    use ElasticQueryBuilderTrait;

    /** @var string */
    protected $language;

    /**
     * CampsiteSearchQueryBuilder constructor.
     * @param int $scopeId
     * @param string $state
     * @param string $language
     */
    public function __construct(
        int $scopeId,
        string $state = null,
        string $language = 'fre'
    ) {
        $this->filters['scopeId'] = $scopeId;
        $this->filters['state'] = $state;
        $this->language = $language;
    }

    /**
     * @param \DateTime|null $date
     * @param string|null $scale
     * @param string $offset
     * @param float|null $decay
     * @return $this
     */
    public function setRecentFirst(
        \DateTime $date = null,
        string $scale = null,
        string $offset = null,
        float $decay = null
    ) {
        $this->filters['recent_first'] = [
            'origin' => ($date ?? date_create())->format('Y-m-d\TH:i:sP'),
            'scale' => $scale ?? '30d',
            'offset' => $offset ?? '60d',
            'decay' => $decay ?? 0.5,
        ];
        return $this;
    }

    /**
     * @param array $profil
     * @return \Cms\Search\Article\ArticleSearchQueryBuilder
     */
    public function setProfil(array $profil)
    {
        $this->filters['profil'] = $profil;
        return $this;
    }

    /**
     * @param \DateTime $date
     * @return $this
     */
    public function setPublishedDate(\DateTime $date)
    {
        $this->filters['published_date'] = $date;
        return $this;
    }

    /**
     * @param array $categoryId
     * @return $this
     */
    public function setCategoryId(array $categoryId)
    {
        $this->filters['category_ids'] = $categoryId;
        return $this;
    }

    /**
     * @return array
     */
    public function getAggsAsArray() : array
    {
        return [];
    }

    /**
     * @return array
     */
    public function getQueryAsArray() : array
    {
        $query = [];
        $query['bool']['filter'][] = $this->getTermQuery(
            'scope_id',
            $this->filters['scopeId']
        );
        if (!empty($this->filters['state'])) {
            $query['bool']['filter'][] = $this->getTermQuery(
                'state',
                $this->filters['state']
            );
        }
        if (!empty($this->filters['profil'])) {
            $query['bool']['filter'][] = $this->getTermsQuery(
                'profil',
                $this->filters['profil']
            );
        }
        if (!empty($this->filters['category_ids'])) {
            $query['bool']['filter'][] = $this->getTermsQuery(
                'category_id',
                $this->filters['category_ids']
            );
        }

        // recherche texte
        if (!empty($this->filters['searchText'])) {
            $query['bool']['must'][] = [
                'multi_match' => [
                    'query' => $this->filters['searchText'],
                    'fields' => ['title^5', 'abstract', 'content'],
                ],
            ];
        }

        // check sur les dates
        if ($this->filters['published_date'] instanceof \DateTime) {
            $query['bool']['filter'][] = $this->getRangeQuery(
                'last_published_at',
                $this->filters['published_date']->format('Y-m-d'),
                $this->filters['published_date']->modify('+ 1 day')->format('Y-m-d'),
                false
            );
        } // application du boost sur les articles recent
        elseif (!empty($this->filters['recent_first'])) {
            $score = [
                'query' => $query,
                'exp' => [
                    'last_published_at' => $this->filters['recent_first'],
                ],
            ];
            // reset query
            $query = [];
            $query['query']['function_score'] = $score;
        }

        return $query;
    }
}
