<?php


namespace Cms\Search\Campsite;

use Cms\LinkResource\LinkResourcePrefsIdentifier;
use Cms\Model\Campsite\CampsiteDistanceType;
use Cms\Model\Campsite\Feature\FeatureCode;
use Cms\Search\ElasticQueryBuilderTrait;

/**
 * Class CampsiteSearchFragmentFactory
 * @package Cms\Search\Campsite
 */
class CampsiteSearchFragmentFactory
{
    use ElasticQueryBuilderTrait;

    /** @var string */
    public $division_nested_path = 'geo_divisions.division_name_i18n';

    /** @var string */
    public $feature_nested_path = 'features_nested';

    /** @var string */
    public $product_nested_path = 'campsite_products_nested';

    /** @var string */
    public $admin_nested_path = 'geo_admins_nested';

    /** @var string */
    public $admin_i18n_nested_path = 'geo_admins_nested.admin_name_i18n';

    /** @var string */
    public $address_nested_path = 'addresses_nested';

    /**
     * @param array|string $countryCode
     * @param string $addressType
     * @param bool $excludeDomTom
     * @return array
     */
    public function getAddressCountryQuery(
        $countryCode,
        string $addressType,
        bool $excludeDomTom = false
    ) : array {
        $nestedPath = $this->address_nested_path;
        $subBoolQuery = [
            $this->getTermsQuery(
                $nestedPath . '.country_code',
                (array)$countryCode
            ),
            $this->getTermQuery(
                $nestedPath . '.type',
                $addressType
            ),
        ];
        if ($excludeDomTom) {
            $subBoolQuery[] = [
                'range' => [
                    $nestedPath . '.admin1_code' => [
                        'lte' => 99,
                    ],
                ],
            ];
        }
        $subQuery = $this->getBoolQuery($subBoolQuery);
        $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
        return $nestedQuery;
    }

    /**
     * @param array $topRight
     * @param array $bottomLeft
     * @param string $addressType
     * @return array
     */
    public function getAddressBoundingBoxFilter(
        array $topRight,
        array $bottomLeft,
        string $addressType = null
    ) : array {
        $subQuery = [
            [
                'term' => [
                    $this->address_nested_path . '.type' => $addressType
                        ?: LinkResourcePrefsIdentifier::ADDRESS_LOC,
                ],
            ],
            [
                'geo_bounding_box' => [
                    $this->address_nested_path . '.geo_location' => [
                        'top_right' => $topRight,
                        'bottom_left' => $bottomLeft,
                    ],
                ],
            ],
        ];
        return $this->getBoolQuery($subQuery);
    }

    /**
     * @param string|null $queryString
     * @param string $addressType
     * @return array
     */
    public function getAddressBoolSearchQuery(
        string $queryString = null,
        string $addressType
    ) : array {
        $nestedPath = $this->address_nested_path;
        $subQuery[] = [
            'term' => [
                $nestedPath . '.type' => $addressType,
            ],
        ];
        if ($queryString !== null && \strlen($queryString) > 0) {
            $subQuery[] = [
                'match' => [
                    $nestedPath . '.city_name_fre' => [
                        'query' => $queryString,
                        'operator' => 'and',
                    ],
                ],
            ];
        }
        return $this->getBoolQuery($subQuery);
    }

    /**
     * @param array $serviceAreas
     * @return array
     */
    public function getServiceAreasSearchQuery(array $serviceAreas) : array
    {
        $subQuery = [];
        foreach ($serviceAreas ?? [] as $areaData) {
            $areaQuery = [];
            if (!empty($areaData[0])) {
                $areaQuery[] = $this->getTermsQuery(
                    'type',
                    $areaData[0]
                );
            }
            if (!empty($areaData[1])) {
                $areaQuery[] = $this->getTermsQuery(
                    'features.feature_code',
                    $areaData[1]
                );
            }
            if ($areaQuery) {
                $subQuery[] = [
                    'bool' => ['must' => $areaQuery],
                ];
            }
        }
        return [
            'bool' => ['should' => $subQuery],
        ];
    }

    /**
     * @param string|null $lowValue
     * @param string|null $highValue
     * @return array
     * @throws \InvalidArgumentException
     */
    public function getTotalEmpRangeSearchQuery($lowValue = null, $highValue = null) : array
    {
        if (!$lowValue && !$highValue) {
            throw new \InvalidArgumentException('lowValue or highValue must be set');
        }
        $nestedPath = $this->feature_nested_path;
        $searchQuery[] = $this->getTermQuery(
            $nestedPath . '.feature_code',
            FeatureCode::NBTOTALEMP
        );
        $range = [];
        if ($highValue) {
            $range['lte'] = (int)$highValue;
        }
        if ($lowValue) {
            $range['gte'] = (int)$lowValue;
        }
        $searchQuery[] = [
            'range' => [
                $nestedPath . '.quantity' => $range,
            ],
        ];
        return $this->getBoolQuery($searchQuery);
    }

    /**
     * @param string|null $dateOpen
     * @param string|null $dateClose
     * @param bool|null $alwaysOpen
     * @return array|null
     */
    public function getPeriodQuery($dateOpen = null, $dateClose = null, bool $alwaysOpen = true)
    {
        if ($dateOpen && $dateClose) {
            return [
                'bool' => [
                    'filter' => [
                        $this->getRangeQuery('campsite_opened_at', $dateOpen),
                        $this->getRangeQuery('campsite_closed_at', null, $dateClose),
                    ],
                ],
            ];
        }
        if ($alwaysOpen === true) {
            return $this->getTermQuery('campsite_always_open', true);
        }
        return null;
    }

    /**
     * @param string|array $featureCode
     * @param array $values
     * @param array|null $distances
     * @return array
     */
    public function getFeatureValueSearchQuery(
        $featureCode,
        array $values = null,
        array $distances = null
    ) : array {
        $nestedPath = $this->feature_nested_path;
        if (!empty($distances)) {
            $searchQuery[] = $this->getTermsQuery(
                $nestedPath . '.distance_type',
                $distances
            );
        }
        if (!empty($values)) {
            $searchQuery[] = $this->getTermsQuery(
                $nestedPath . '.value',
                $values
            );
        }
        $searchQuery[] = $this->getTermsQuery(
            $nestedPath . '.feature_code',
            (array)$featureCode
        );
        if (count($searchQuery) > 1) {
            return $this->getBoolQuery($searchQuery);
        }
        return $searchQuery[0];
    }

    /**
     * @param array $productIds
     * @param array|null $quantityRange [lt, gt, lteq, gteq, boost]
     * @param string $boolOperator
     * @return array
     */
    public function getProductIdsSearchQuery(
        array $productIds,
        array $quantityRange = null,
        string $boolOperator = 'must'
    ) : array {
        $searchQueryCodes = [];

        foreach ($productIds as $productId) {
            $searchQueryCodes[] = $this->getNestedQuery(
                $this->product_nested_path,
                $this->getTermsQuery(
                    $this->product_nested_path . '.id',
                    (array)$productId
                )
            );
        }

        if (!empty($searchQueryCodes)) {
            if (count($searchQueryCodes) > 1) {
                $searchQuery = $this->getBoolQuery($searchQueryCodes, $boolOperator);
            } else {
                $searchQuery = $searchQueryCodes[0];
            }
            if (!empty($quantityRange)) {
                $searchQuery = $this->getBoolQuery([
                    $searchQuery,
                    $this->getNestedQuery(
                        $this->product_nested_path,
                        $this->getRangeQuery(
                            $this->product_nested_path . '.quantity',
                            ...$quantityRange
                        )
                    ),
                ]);
            }
            return $searchQuery;
        }
        return [];
    }

    /**
     * @param array $featureCodes
     * @param array|null $commonValues
     * @param array|null $commonDistance
     * @param string $boolOperator
     * @return array
     */
    public function getFeatureCodeSearchQuery(
        array $featureCodes,
        array $commonValues = null,
        array $commonDistance = null,
        $boolOperator = 'must'
    ) : array {
        $nestedPath = $this->feature_nested_path;
        foreach ($featureCodes as $code) {
            $searchQueryCodes[] = $this->getNestedQuery(
                $nestedPath,
                $this->getFeatureValueSearchQuery(
                    $code,
                    $commonValues,
                    $commonDistance
                )
            );
        }
        if (!empty($searchQueryCodes)) {
            if (count($searchQueryCodes) > 1) {
                return $this->getBoolQuery($searchQueryCodes, $boolOperator);
            }
            return $searchQueryCodes[0];
        }
        return [];
    }

    /**
     * @return array
     */
    public function getPoolSearchQuery() : array
    {
        return $this->getFeatureValueSearchQuery(
            [FeatureCode::PISCINE_CHAUF, FeatureCode::PISCINE_COUV, FeatureCode::PISCINE_PLEINAIR],
            ['1'],
            [CampsiteDistanceType::DIRECT, CampsiteDistanceType::UNDEFINED]
        );
    }

    /**
     * @param string $language
     * @param string|null $queryString
     * @return array
     */
    public function getAdminSearchQuery(string $language, string $queryString = null) : array
    {
        $nestedPath = $this->admin_i18n_nested_path;
        $subQuery[] = [
            'term' => [
                $nestedPath . '.language' => $language,
            ],
        ];
        if ($queryString !== null && \strlen($queryString) > 0) {
            $subQuery[] = [
                'match' => [
                    $nestedPath . '.admin_name' => $queryString,
                ],
            ];
        }
        return $this->getBoolQuery($subQuery);
    }

    /**
     * @param string $language
     * @param string|null $queryString
     * @return array
     */
    public function getDivisionSearchQuery(string $language, string $queryString = null) : array
    {
        $nestedPath = $this->division_nested_path;
        $subQuery[] = [
            'term' => [
                $nestedPath . '.language' => $language,
            ],
        ];
        if ($queryString !== null && \strlen($queryString) > 0) {
            $subQuery[] = [
                'match' => [
                    $nestedPath . '.division_name' => $queryString,
                ],
            ];
        }
        return $this->getBoolQuery($subQuery);
    }

    /**
     * @return array
     */
    public function getAggsAsArray() : array
    {
        return [];
    }

    /**
     * @return array
     */
    public function getQueryAsArray() : array
    {
        return [];
    }
}
