<?php


namespace Cms\LinkResource\Provider;

use Cms\LinkResource\LinkResourceType;

/**
 * Class AbstractProviderDelegator
 * @package Cms\LinkResource\Provider
 */
abstract class AbstractProviderDelegator extends AbstractProviderAdapter implements
    SearchAdapterAwareInterface,
    DataAdapterAwareInterface,
    RecursiveAdapterAwareInterface
{
    use SearchAdapterAwareTrait;

    /** @var  SearchAdapterAwareInterface[] */
    protected $searchAdapters;

    /** @var  DataAdapterAwareInterface[] */
    protected $dataAdapters;

    /** @var  \Cms\LinkResource\Provider\RecursiveAdapterAwareInterface[] */
    protected $recursiveAdapters;

    /**
     * Délégue le traitement de recherhce/getData a un provider "enfant"
     * @param DataAdapterAwareInterface|DelegateTypeAdapterInterface|SearchAdapterAwareInterface $adapter
     */
    public function delegate(DelegateTypeAdapterInterface $adapter)
    {
        foreach ($adapter->getTypes() as $typeCode) {
            if ($adapter instanceof SearchAdapterAwareInterface) {
                $this->searchAdapters[(string)$typeCode] = $adapter;
            }
            if ($adapter instanceof DataAdapterAwareInterface) {
                $this->dataAdapters[(string)$typeCode] = $adapter;
            }
            if ($adapter instanceof RecursiveAdapterAwareInterface) {
                $this->recursiveAdapters[(string)$typeCode] = $adapter;
            }
            // ajout du typeCode a la liste actuel
            if (!\in_array((string)$typeCode, $this->types)) {
                $this->types[] = (string)$typeCode;
            }
        }
    }

    /**
     * @inheritdoc
     */
    public function getResources($typeCode, $objectId) : array
    {
        // on délégue l'execution de la recherche a un des adapter fourni
        $result = [];
        if (!empty($this->recursiveAdapters[(string)$typeCode])) {
            try {
                $result = $this->recursiveAdapters[(string)$typeCode]->getResources($typeCode, $objectId);
            } catch (\Exception $e) {
                trigger_error(
                    "fetch resources of $typeCode $objectId failed : " . $e,
                    E_USER_WARNING
                );
            }
        }
        return $result;
    }

    /**
     * @param string|LinkResourceType $typeCode
     * @param string $query
     * @return array
     * @throws \InvalidArgumentException
     */
    public function execSearch($typeCode, $query) : array
    {
        // On checke qu'on cherche bien une query
        if (empty($query)) {
            throw new \InvalidArgumentException('query is missing');
        }

        // on délégue l'execution de la recherche a un des adapter fourni
        $result = [];
        if (!empty($this->searchAdapters[(string)$typeCode])) {
            $result = $this->searchAdapters[(string)$typeCode]->search($typeCode, $query);
        }
        return $result;
    }

    /**
     * @param string|LinkResourceType $typeCode
     * @param int $objectId
     * @return mixed
     */
    public function getData($typeCode, $objectId)
    {
        // on délégue l'execution de la recherche a un des adapter fourni
        $result = null;
        if (!empty($this->dataAdapters[(string)$typeCode])) {
            try {
                $result = $this->dataAdapters[(string)$typeCode]->getData($typeCode, $objectId);
            } catch (\Exception $e) {
                trigger_error(
                    "fetch model of $typeCode $objectId failed : " . $e,
                    E_USER_WARNING
                );
            }
        }
        return $result;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param int $objectId
     * @return string
     */
    public function getLabel($typeCode, $objectId) : string
    {
        // on délégue l'execution de la recherche a un des adapter fourni
        $result = '';
        if (!empty($this->dataAdapters[(string)$typeCode])) {
            try {
                $result = $this->dataAdapters[(string)$typeCode]->getLabel($typeCode, $objectId);
            } catch (\Exception $e) {
                trigger_error(
                    "fetch label of $typeCode $objectId failed : " . $e,
                    E_USER_WARNING
                );
            }
        }
        return $result ?: '';
    }
}
