<?php

namespace Cms\Search\Article;

use Cms\DashParser\DashParser;
use Cms\DashParser\DashParserAwareTrait;
use Cms\DashParser\Serializer\HtmlSerializer;
use Cms\Model\Article\ArticleModel;
use Elastica\Document;
use Move\I18n\I18nLanguageCode;
use Move\Search\Analyzer\FrenchHtmlTextAnalyzer;
use Move\Search\Analyzer\FrenchSearchAnalyzer;
use Move\Search\Analyzer\FrenchTextAnalyzer;
use Move\Search\Analyzer\SearchWordAnalyzer;
use Move\Search\Analyzer\WordAnalyzer;
use Move\Search\ElasticMappingBuilder;
use Move\Search\Mapping\ElasticMappingProperty;
use Move\Search\Mapping\KeywordMapping;
use Move\Search\Mapping\ObjectMapping;
use Move\Search\Mapping\TextMapping;

/**
 * Class ArticleIndexFactory
 * @package Cms\Search\Article
 */
class ArticleIndexFactory
{
    use DashParserAwareTrait;

    /** Nom du type pour les articles dans l'index */
    public const MAPPING_NAME = 'cms_article';

    /**
     * @return \Move\Search\ElasticMappingBuilder
     */
    public static function createMappingBuilder() : ElasticMappingBuilder
    {
        $builder = new ElasticMappingBuilder(self::MAPPING_NAME, [
            new ElasticMappingProperty('id', 'long'),
            new ElasticMappingProperty('scope_id', 'long'),
            new ElasticMappingProperty('container_id', 'long'),
            new ElasticMappingProperty('category_id', 'long'),

            new ElasticMappingProperty('cms_user_id', 'long', false),

            new ElasticMappingProperty('cover_flag', 'boolean'),

            new KeywordMapping('tags', false),
            new KeywordMapping('state'),
            new KeywordMapping('profil'),
            new KeywordMapping('parution_num'),
            new KeywordMapping('language'),

            new ObjectMapping('i18n', array_map(static function ($lng) {
                return new ObjectMapping($lng, [
                    (new TextMapping('title'))->withAnalyzer(
                        new FrenchTextAnalyzer(),
                        new FrenchSearchAnalyzer()
                    ),

                    (new TextMapping('abstract'))->withAnalyzer(
                        new FrenchHtmlTextAnalyzer(),
                        new FrenchSearchAnalyzer()
                    ),

                    (new TextMapping('content'))->withAnalyzer(
                        new FrenchHtmlTextAnalyzer(),
                        new FrenchSearchAnalyzer()
                    ),
                ]);
            }, I18nLanguageCode::toArray())),

            $dateProp = new ElasticMappingProperty('created_at', 'date'),
            $dateProp->withPropName('modified_at'),

            new ElasticMappingProperty('last_published_at', 'date'),

            (new TextMapping('author_signature'))->withAnalyzer(
                new WordAnalyzer(),
                new SearchWordAnalyzer()
            ),

            (new TextMapping('title'))->withAnalyzer(
                new FrenchTextAnalyzer(),
                new FrenchSearchAnalyzer()
            ),

            (new TextMapping('abstract'))->withAnalyzer(
                new FrenchHtmlTextAnalyzer(),
                new FrenchSearchAnalyzer()
            ),

            (new TextMapping('content'))->withAnalyzer(
                new FrenchHtmlTextAnalyzer(),
                new FrenchSearchAnalyzer()
            ),
        ]);
        return $builder;
    }

    /**
     * @param ArticleModel $article
     * @param array $tags
     * @param array $i18n
     * @return \Elastica\Document
     */
    public static function createDocument(ArticleModel $article, array $tags = [], array $i18n = []) : Document
    {
        $languages = array_map(static function ($item) {
            return $item['language'];
        }, $i18n);
        $serializer = new self();
        $articleDocument = [
            'id' => $article->id,
            'parution_num' => $article->parution_num,
            'container_id' => $article->container_id,
            'cms_user_id' => $article->cms_user_id,
            'scope_id' => $article->scope_id,
            'category_id' => [$article->category_id],
            'cover_flag' => $article->cover_flag,
            'title' => $article->title,
            'author_signature' => $article->author_signature,
            'profil' => (string)$article->profil,
            'abstract' => $serializer->serializeContent($article->abstract),
            'content' => $serializer->serializeContent($article->content),
            'state' => (string)$article->state,
            'tags' => array_values($tags),
            'i18n' => array_combine($languages, array_map(static function ($item) use ($serializer) {
                return [
                    'title' => $item->title,
                    'abstract' => $serializer->serializeContent($item->abstract),
                    'content' => $serializer->serializeContent($item->content),
                ];
            }, $i18n)),
            'language' => array_map(static function ($item) {
                return $item['language'];
            }, $i18n),
            'created_at' => $article->created_at->format(DATE_W3C),
            'last_published_at' => $article->last_published_at ? $article->last_published_at->format(DATE_W3C) : null,
            'modified_at' => $article->modified_at->format(DATE_W3C),
        ];

        // charge fre si rien
        if (empty($articleDocument['language'])
            || !\in_array(I18nLanguageCode::FR, $articleDocument['language'], true)
        ) {
            $articleDocument['language'][] = I18nLanguageCode::FR;
        }

        // envoi des données
        $document = new Document($article->id, $articleDocument, self::MAPPING_NAME);
        return $document;
    }

    /**
     * @param DashParser $parser
     * @return string
     */
    protected function serializeParser(DashParser $parser) : string
    {
        $serializer = new HtmlSerializer(
            $parser->getEntityCollection()
        );
        return $serializer->serialize($parser->getBlocksCollection());
    }
}
