<?php

namespace Cms\Client\Command\Article;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Article\Model\ArticleRestfulModel;
use Cms\Client\Command\Article\Command\FetchAllArticleByFilterCommand;
use Cms\Client\Command\Article\Command\FetchArticleCommand;
use Cms\Client\Command\CommandExceptionHandleTrait;
use Cms\Client\Service\ArticleService;
use Cms\Model\Article\ArticleState;
use Move\Cache\CacheAwareInterface;
use Move\Command\Command\FetchBulkCommand;
use Move\Command\Exception\BadParameterException;
use Move\Command\Exception\NotFoundException;
use Move\Command\Exception\OutOfBoundsException;
use Move\Http\Client\Exception\HttpRequestException;
use Move\I18n\I18nLanguageCode;
use Psr\Cache\CacheItemInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class ArticleCommandHandler
 * @package AutoMag\Command\Article
 */
class ArticleCommandHandler implements LoggerAwareInterface, CacheAwareInterface
{
    use CacheHelperTrait;
    use LoggerAwareTrait;
    use CommandExceptionHandleTrait;

    /**
     * @var ArticleService
     */
    private $articleService;

    /**
     * ArticleCommandHandler constructor.
     * @param ArticleService $articleService
     */
    public function __construct(ArticleService $articleService)
    {
        $this->articleService = $articleService;
    }

    /**
     * @param \Move\Command\Command\FetchBulkCommand $command
     * @return ArticleRestfulModel[]
     * @throws \Move\Command\Exception\NotFoundException
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public function handleFetchBulk(FetchBulkCommand $command) : array
    {
        $articleService = $this->articleService;
        $articleClient = $articleService->getArticleClient();
        try {
            $articles = $articleClient->fetchBulkItems($command->getIds());
            if (count($articles) !== count($command)) {
                $this->logger->warning((count($articles) . '/' . count($command)) . ' article was not found');
            }
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('ArticleCmdHd_FetchBulk', $e);
        }
        return $articles ?? [];
    }

    /**
     * @param FetchAllArticleByFilterCommand $command
     * @return ArticleRestfulModel[]|\Iterator
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\OutOfBoundsException
     */
    public function handleFetchAllArticleByFilter(FetchAllArticleByFilterCommand $command)
    {
        $pageNum = $command->getPageNum();
        $lexiqueId = $command->getLexiqueWordId();
        $scopeId = $command->getScopeId();
        $categoryId = $command->getCategoryIds() ?: [];
        $profils = $command->getProfils();
        $sinceDate = $command->getSinceDate();
        $tags = $command->getTags();
        $lang = $command->getLang();
        $cmsUserId = $command->getCmsUserId();

        // récuperation des catégories enfants
        $categoryParentId = $command->getParentCategoryId();
        if (!empty($categoryParentId)) {
            if (!empty($categoryId) && !\is_array($categoryId)) {
                $categoryId = [$categoryId];
            }
            if (!\is_array($categoryParentId)) {
                $categoryParentId = [$categoryParentId];
            }
            $categoryClient = $this->articleService->getCategoryClient();
            foreach ($categoryParentId as $id) {
                $categories = $categoryClient->getIndexByParentId($id, -1);
                foreach ($categories as $childCategory) {
                    $categoryId[] = $childCategory->id;
                }
            }
        }

        // récuperation des articles
        $articleClient = $this->articleService->getArticleClient();
        if (!empty($categoryId)) {
            $articleBuilder = $articleClient->getBuilder()
                ->setCategories($categoryId);
        } elseif (!empty($scopeId)) {
            $articleBuilder = $articleClient->getBuilder()
                ->setScope($scopeId);
        } else {
            throw new BadParameterException('scope_id and category_id are empty');
        }

        // récuperation avec date limite
        if ($sinceDate instanceof \DateTime) {
            $articleBuilder->setSinceDate($sinceDate);
        }

        if (!empty($lexiqueId)) {
            $articleBuilder->setLexiqueWordId($lexiqueId);
        }

        if (!empty($tags)) {
            $articleBuilder->setTags($tags);
        }

        if (!empty($lang) && $lang !== I18nLanguageCode::FR) {
            $articleBuilder->setLang($lang);
        }

        if (!empty($cmsUserId)) {
            $articleBuilder->setCmsUserId($cmsUserId);
        }

        // récuperation avec link resource
        if (!empty($command->getLinkResourceType()) && !empty($command->getLinkResourceRef())) {
            $articleBuilder->setLinkResource(
                $command->getLinkResourceType(),
                $command->getLinkResourceRef()
            );
        }

        if (!$command->getCoverFirst() && !$command->isCoverProfil()) {
            $articleBuilder->setState($command->getState() ?: ArticleState::PUBLISH);
        } elseif ($command->isCoverProfil()) {
            $articleBuilder->setState($command->getState() ?: ArticleState::PUBLISH);
            $articleBuilder->hasCoverProfil(true);
        } else {
            $articleBuilder->hasCoverFirst(true);
        }

        if (($coverListId = $command->getCoverList())) {
            $articleBuilder->setCoverList($coverListId);
        }

        if ($command->isSortByFirstPublished()) {
            $articleBuilder->setSortCol('first_published_at', $command->getSortByFirstPublished());
            $articleBuilder->setSortCol('modified_at', $command->getSortByFirstPublished());
        }

        try {
            $articles = $articleBuilder
                ->setProfils($profils)
                ->getIndex($pageNum);
        } catch (HttpRequestException $e) {
            if ($e->getMessage() === 'page out of bounds') {
                throw new OutOfBoundsException($e->getMessage(), 0, $e);
            }
            $this->logHttpRequestException('ArticleCmdHd_FetchAllArticleByFilter', $e);
        } finally {
            if (!isset($articles)) {
                $articles = null;
            }
        }
        return $articles;
    }

    /**
     * @param FetchArticleCommand $command
     * @return ArticleRestfulModel|null
     */
    public function handleFetchArticle(FetchArticleCommand $command)
    {
        $articleId = $command->getId();
        $articleSlug = $command->getSlug();
        $language = $command->getLanguage();
        $scopeId = $command->getScopeId();

        $cacheKey = $scopeId . '^article^' . ($articleId ?: $articleSlug) . '^' . $language;

        /** @var CacheItemInterface $cacheItem */
        if ($this->hitFromCache($cacheKey, 120, $cacheItem)) {
            $articleModel = $cacheItem->get();
        } else {
            // récuperation de l'article
            $articleClient = $this->articleService->getArticleClient();
            try {
                if ($articleId) {
                    $articleModel = $articleClient->fetchItem($articleId);
                } elseif ($articleSlug) {
                    $articleModel = $articleClient->fetchItemBySlug($articleSlug, $scopeId, $language);
                } else {
                    throw new BadParameterException('articleId/articleSlug manquant');
                }
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('ArticleCmdHd_FetchArticle', $e);
            } finally {
                if (!isset($articleModel)) {
                    $articleModel = null;
                }
            }

            // prise en charge du cache
            $this->setInCache($cacheItem, $articleModel);
        }

        if (!$articleModel || (string)$articleModel->state !== ArticleState::PUBLISH) {
            throw new NotFoundException('id:' . ($articleId ?: $articleSlug));
        }
        if ($scopeId && (int)$articleModel->scope_id !== $scopeId) {
            throw new NotFoundException('bad scope:' . ($articleId ?: $articleSlug) . ' >> ' . $scopeId);
        }
        return $articleModel;
    }
}
