<?php

namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\ApiFailedException;
use Cms\Bundle\Member\Exception\FieldException;
use Cms\Bundle\Member\Exception\MemberNotFoundException;
use Cms\Bundle\Member\Exception\RuntimeException;
use Cms\Client\Member\Model\MemberRestfulModel;
use League\Route\Http\Exception\NotFoundException;
use Move\Http\ResponseUtils;
use Move\Http\Strategy\DataContainer\EngineDataContainer;
use Move\Http\Strategy\ResponseFactory;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class AbstractEditMemberController
 * @package Cms\Bundle\Member
 */
abstract class AbstractEditMemberController extends AbstractSaveMemberController
{
    use MemberUtilsTrait;

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @return \Cms\Client\Member\Model\MemberRestfulModel|null
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \League\Route\Http\Exception\NotFoundException
     */
    protected function getMemberWithToken(ServerRequestInterface $request)
    {
        $memberId = null;
        // vérification du token
        if (!$this->checkToken($request, $memberId)) {
            return null;
        }

        // récupération des infos du membre connecté
        try {
            $memberModel = $this->getMemberModel(
                $this->getLoginService()->getMemberClient(),
                $memberId,
                true
            );
        } catch (MemberNotFoundException $e) {
            throw new NotFoundException('member not found', $e);
        }
        return $memberModel;
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Move\Http\Strategy\ResponseFactory|\Psr\Http\Message\ResponseInterface
     * @throws \InvalidArgumentException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \League\Route\Http\Exception\NotFoundException
     */
    public function index(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        $memberModel = $this->getMemberWithToken($request);
        if ($memberModel === null) {
            $redirUrl = $this->getNotAuthorizedUrl();
            return ResponseUtils::withNoCache($response
                ->withStatus(302)
                ->withHeader('location', $redirUrl));
        }

        $data = $this->getEngineContainer($memberModel);
        $factory = new ResponseFactory($data, 200, $response->getHeaders());
        $factory->mergeHeaders(ResponseUtils::getNoCacheHeaders());
        return $factory;
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Move\Http\Strategy\ResponseFactory|\Psr\Http\Message\ResponseInterface
     * @throws \InvalidArgumentException
     * @throws \League\Route\Http\Exception\NotFoundException
     * @throws \Cms\Bundle\Member\Exception\PasswordFailedException
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     * @throws \Cms\Bundle\Member\Exception\FieldException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     */
    public function record(
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        $memberModel = $this->getMemberWithToken($request);
        if ($memberModel === null) {
            $redirUrl = $this->getNotAuthorizedUrl();
            return ResponseUtils::withNoCache($response
                ->withStatus(302)
                ->withHeader('location', $redirUrl));
        }

        $this->validateRequest($request);

        // mise a jour du mot de passe
        if (true === $this->hasUpdatePassword($request, $newpass)) {
            $memberModel->update_password = $newpass;
            $memberModel->password_scope_id = $this->getLoginService()->getScopeId();
        }

        // filtrage des data en params
        $dataset = $request->getParsedBody();
        $params = $this->filterRequestForParams($dataset);

        // changement d'email => envoi de mail ?
        if ($memberModel->email !== $dataset['email']) {
            // vérification d'unicité
            try {
                $memberModelForEmail = $this->getMemberModel(
                    $this->getLoginService()->getMemberClient(),
                    $dataset['email'],
                    true
                );
                if ($memberModelForEmail) {
                    throw new FieldException(
                        'email',
                        MemberErrorCode::EMAIL_ALREADY_EXIST
                    );
                }
            } catch (MemberNotFoundException $e) {
            }
            $memberModel->email = $dataset['email'];
        }

        // ajout des attributs
        $scopeId = $this->getLoginService()->getScopeId();
        foreach ($params as $attr => $val) {
            $memberModel->setAttributes($scopeId, $attr, $val, true);
        }

        // record member
        $memberModel = $this->handleMemberRecord($memberModel);

        $data = $this->getEngineContainer($memberModel);
        return new ResponseFactory($data, 200, $response->getHeaders());
    }

    /**
     * @param \Cms\Client\Member\Model\MemberRestfulModel $memberModel
     * @return MemberRestfulModel
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     */
    protected function handleMemberRecord(MemberRestfulModel $memberModel)
    {
        try {
            // sauvegarde du membre
            $this->getLoginService()->getMemberClient()->saveItem($memberModel);
            return $memberModel;
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot update user information with API',
                $e,
                MemberErrorCode::SAVE_MEMBER_API_FAILED
            );
        }
    }

    /**
     * Défini si un nouveau mot de passe a été demandé par le client, si oui
     * on renvoi true, et la variable $newpass doit être renseigné
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param string $newpass
     * @return bool|string
     * @throws \Cms\Bundle\Member\Exception\FieldMissingException
     * @throws \Cms\Bundle\Member\Exception\PasswordFailedException
     */
    protected function hasUpdatePassword(
        ServerRequestInterface $request,
        &$newpass
    ) : bool {
        $dataset = $request->getParsedBody();
        $this->validatePassword($dataset, 'password', 'confirm_password', false);
        if (!empty($dataset['password'])) {
            $newpass = $dataset['password'];
            return true;
        }
        return false;
    }

    /**
     * Renvoi l'url de redirection en cas de deconnexion du membre
     * @return string
     */
    abstract protected function getNotAuthorizedUrl() : string;

    /**
     * Renvoi le formulaire d'edition du compte
     * @param \Cms\Client\Member\Model\MemberRestfulModel $memberModel
     * @param \Cms\Bundle\Member\Exception\RuntimeException|null $error
     * @return \Move\Http\Strategy\DataContainer\EngineDataContainer
     */
    abstract protected function getEngineContainer(
        MemberRestfulModel $memberModel = null,
        RuntimeException $error = null
    ) : EngineDataContainer;
}
