<?php

namespace Cms\Route;

use Move\Http\Route\AbstractRouteProvider;

/**
 * Class AbstractRoute
 * @package Cms\Route
 */
abstract class AbstractRoute extends AbstractRouteProvider
{

    /** @var array */
    protected $routes = [];

    /** @var callable[] */
    private $decorators;

    /** @var \Cms\Route\PathBuilder[] */
    private $pathBuilders;

    /**
     * @param string $path
     * @param string $handlerClass
     * @param string|array $method
     * @param string $defaultAction
     * @return \League\Route\Route
     */
    protected function addPageRoute($path, $handlerClass, $method = 'get', $defaultAction = 'index')
    {
        $methods = (array)$method;
        $methods = array_map('strtoupper', $methods);
        $path = $this->routes[$path] ?? $path;
        return $this->router->map($methods, $path, $handlerClass . '::' . $defaultAction);
    }

    /**
     * function can return string or builder
     * ```
     * function ($identifier, PathBuilder $builder) {
     *      $builder->addParam('name', 'value');
     *      return $builder;
     * }
     * ```
     * @param callable $decorator
     * @param string|null $identifier
     * @return $this
     */
    public function setBuilderDecorator(callable $decorator, string $identifier = null)
    {
        $identifier = $identifier ?: 'default';
        $this->decorators[$identifier] = $decorator;
        return $this;
    }

    /**
     * @param string|null $identifier
     * @return callable|null
     */
    public function getBuilderDecorator(string $identifier = null)
    {
        $identifier = $identifier ?: 'default';
        if ($identifier !== 'default'
            && !isset($this->decorators[$identifier])
        ) {
            $identifier = 'default';
        }
        return $this->decorators[$identifier] ?? null;
    }

    /**
     * Si n'existe pas on les crées
     * @param string $identifier
     * @return \Cms\Route\PathBuilder
     */
    public function getPathBuilder(string $identifier) : PathBuilder
    {
        if (!isset($this->pathBuilders[$identifier])) {
            /** @noinspection ExceptionsAnnotatingAndHandlingInspection */
            $this->registerPathBuilder($identifier);
        }
        return $this->pathBuilders[$identifier];
    }

    /**
     * @param string $identifier
     * @param callable|null $decorator
     * @return callable
     */
    public function prepareBuilder(
        string $identifier,
        callable $decorator = null
    ) : callable {
        // renseigne le decorateur pour cet identifier
        if ($decorator !== null) {
            $this->setBuilderDecorator($decorator, $identifier);
        }
        $builder = $this->getPathBuilder($identifier);

        // application du decorateur
        return function () use ($builder, $identifier) {
            $decorator = $this->getBuilderDecorator($identifier);
            $oriBuilder = clone $builder;
            if (\is_callable($decorator)) {
                $decorate = $decorator($identifier, $oriBuilder);
                if ($decorate instanceof PathBuilder) {
                    return $decorate->build();
                }
                if (\is_string($decorate)) {
                    return $decorate;
                }
            }
            return $oriBuilder->build();
        };
    }

    /**
     * @param string $identifier
     * @param array|\Cms\Route\PathBuilder $init overwrite les routeParams existante si non vide
     * @param bool $overwrite si false on renvoi une exception si les routeParams existe
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function registerPathBuilder(
        string $identifier,
        $init = null,
        bool $overwrite = true
    ) {
        if (!$overwrite && isset($this->pathBuilders[$identifier])) {
            throw new \InvalidArgumentException('route builder already exist for identifier ' . $identifier);
        }
        if (!isset($this->pathBuilders[$identifier]) || $overwrite) {
            // récuperartion de la route
            $path = $this->routes[$identifier] ?? $identifier;
            if ($init instanceof PathBuilder) {
                $this->pathBuilders[$identifier] = $init;
            } else {
                $pathBuilder = new PathBuilder($path);
                $this->pathBuilders[$identifier] = $pathBuilder;
            }
        }
        if (\is_array($init)) {
            $this->pathBuilders[$identifier]->mergeParams($init);
        }
        return $this;
    }

    /**
     * @param string $identifier
     * @param callable|null $decorator
     * @return null|string
     */
    public function buildPath(
        string $identifier,
        callable $decorator = null
    ) {
        $path = \call_user_func($this->prepareBuilder($identifier, $decorator));
        return $path ? (string)$path : null;
    }

    /**
     * @param string $identifier
     * @param array|null $userData
     * @return null|string
     */
    public function buildWithArray(string $identifier, array $userData = null)
    {
        $identifier = $identifier ?: 'default';
        /** @noinspection ExceptionsAnnotatingAndHandlingInspection */
        return $this
            ->registerPathBuilder($identifier, $userData)
            ->buildPath($identifier);
    }

    /**
     * @deprecated
     * @param string $identifier
     * @param array $userData
     * @return string|null
     */
    public function getPathByArray($identifier, array $userData = [])
    {
        $identifier = $identifier ?: 'default';
        return $this->buildWithArray($identifier, $userData);
    }
}
