<?php

namespace Move\Template\Html\Meta;

/**
 * Class SeoBundle
 * @package Move\Template\Html\Meta
 */
class SeoBundle implements MetaProviderInterface
{

    /** @var int */
    public static $DESC_LIMIT = 320;

    /** @var \Move\Template\Html\Meta\Title */
    private $title;

    /** @var \Move\Template\Html\Meta\Name */
    private $description;

    /** @var \Move\Template\Html\Meta\Url */
    private $canonicalUrl;

    /** @var \Move\Template\Html\Meta\Property */
    private $ogDescription;

    /** @var \Move\Template\Html\Meta\Property */
    private $ogTitle;

    /**
     * SeoBundle constructor.
     * @param string $title
     * @param string $description
     * @param string $canonicalUrl
     */
    public function __construct($title, $description, $canonicalUrl)
    {
        $description = strip_tags($description);
        $suffix = mb_strlen($description) > static::$DESC_LIMIT ? '...' : '';
        $description = mb_substr($description, 0, static::$DESC_LIMIT) . $suffix;

        $this->title = new Title($title);
        $this->ogTitle = new Property('og:title', $title);
        $this->description = new Name('description', $description);
        $this->ogDescription = new Property('og:description', $description);
        $this->canonicalUrl = new Url($canonicalUrl);
    }

    /**
     * Retourne un identifiant
     * @return string
     */
    public function ident()
    {
        return 'SeoBundle';
    }

    /**
     * Renvoi la copie sous forme de tableau de la metadata
     * @return array
     */
    public function getArrayCopy()
    {
        return [
            'title' => $this->title->getArrayCopy(),
            'ogTitle' => $this->ogTitle->getArrayCopy(),
            'description' => $this->description->getArrayCopy(),
            'ogDescription' => $this->ogDescription->getArrayCopy(),
            'url' => $this->canonicalUrl->getArrayCopy(),
        ];
    }

    /**
     * Effectue un rendu de l'objet en HTML
     * @return string
     */
    public function __toString()
    {
        return $this->render();
    }

    /**
     * Effectue un rendu de l'objet en HTML
     * @return string
     */
    public function render()
    {
        $content = $this->title->render();
        $content .= $this->description->render();
        $content .= $this->canonicalUrl->render();
        $content .= $this->ogTitle->render();
        $content .= $this->ogDescription->render();
        return $content;
    }

    /**
     * @return Url
     */
    public function getCanonicalUrl()
    {
        return $this->canonicalUrl;
    }

    /**
     * @param MetaCollection $collection
     * @return void
     */
    public function register(MetaCollection $collection)
    {
        $collection->addMulti([
            $this->title,
            $this->description,
            $this->canonicalUrl,
            $this->ogTitle,
            $this->ogDescription,
        ], [], true);
    }
}
