<?php

namespace Move\Http\Strategy;

use Move\Http\Strategy\DataContainer\DataContainer;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

/**
 * Class ResponseFactoryTest
 * @package Move\Http\Strategy
 */
class ResponseFactoryTest extends TestCase
{

    /** @var array */
    protected $dataset = [
            'data' => [1, 2, 3],
            'template' => ''
        ];

    /**
     * Create a mock instance for ResponseInterface
     * @return \PHPUnit_Framework_MockObject_MockObject|ResponseInterface
     */
    public function getMockResponseInterface() {
        // Prepare stream interface
        $streamInterface = $this->getMockBuilder(StreamInterface::class)
            ->getMock();
        $streamInterface
            ->expects($this->any())
            ->method('write')
            ->willReturnSelf();

        // prepare mock response
        $mockResponse = $this->getMockBuilder(ResponseInterface::class)
            ->getMock();

        $mockResponse
            ->method('getBody')
            ->willReturn($streamInterface);

        $mockResponse
            ->method('withStatus')
            ->willReturnSelf();

        $mockResponse
            ->method('withHeader')
            ->willReturnSelf();

        return $mockResponse;
    }

    /**
     * @covers ResponseFactory::__construct
     * @return ResponseFactory
     */
    public function testConstruct()
    {
        $factory = new ResponseFactory($this->dataset);

        $this->assertInstanceOf(ResponseFactory::class, $factory);
        return $factory;
    }

    /**
     * @covers ResponseFactory::determineResponse
     * @depends testConstruct
     * @param ResponseFactory $responseFactory
     */
    public function testDetermineResponse(ResponseFactory $responseFactory)
    {
        $responseInterface = $this->getMockResponseInterface();

        $result = $responseFactory->determineResponse(function(){return '';}, $responseInterface);
        $this->assertNotNull($result);
    }

    /**
     * @covers ResponseFactory::setStatusCode
     * @depends testConstruct
     * @param ResponseFactory $responseFactory
     */
    public function testSetStatusCode(ResponseFactory $responseFactory)
    {
        $result = $responseFactory->setStatusCode(500);
        $this->assertSame($responseFactory, $result);
    }

    /**
     * @covers ResponseFactory::mergeHeaders
     * @depends testConstruct
     * @param ResponseFactory $responseFactory
     */
    public function testMergeHeaders(ResponseFactory $responseFactory)
    {
        $result = $responseFactory->mergeHeaders(['x-auth' => uniqid()]);
        $this->assertSame($responseFactory, $result);
    }


    /**
     * @covers ResponseFactory::getDataset
     * @depends testConstruct
     * @param ResponseFactory $responseFactory
     * @return ResponseFactory
     */
    public function testGetDataset(ResponseFactory $responseFactory)
    {
        $dataset = $responseFactory->getDataset();

        $this->assertInstanceOf(DataContainer::class, $dataset);

        $this->assertEquals($this->dataset, $dataset->getDataset());

        return $responseFactory;
    }

    /**
     * @covers ResponseFactory::setDataset
     * @depends testGetDataset
     * @param ResponseFactory $responseFactory
     */
    public function testSetDataset(ResponseFactory $responseFactory)
    {
        $dataset = ['data' => ''];
        $responseFactory->setDataset($dataset);
        $this->assertEquals($dataset, $responseFactory->getDataset()->getDataset());
    }
}
