<?php

namespace Move\Newsletter\Provider;

use GuzzleHttp\Client;
use Move\Http\Client\GuzzleClientAdapter;
use Move\Http\Client\HttpRequestOptions;
use PHPUnit\Framework\TestCase;

/**
 * Class SarbacaneAdapterTest
 * @package Move\Newsletter\Provider
 */
class SarbacaneAdapterTest extends TestCase
{

    /** @var  GuzzleClientAdapter */
    protected $client;

    /** @var  string */
    protected $email;

    /**
     *
     */
    protected function setUp()
    {
        $guzzle = new Client();
        $this->client = new GuzzleClientAdapter($guzzle, 'api-sd.sarbacane.com', [
            HttpRequestOptions::SECURE => true,
        ]);
        $this->email = 'nicolas.levee@movepublishing.com';
    }

    /**
     * @return array
     */
    public function provideAdapterLogin()
    {
        return [
            [
                'accountId' => '57861',
                'listId' => '5fRi6V3LwkuEeHvDPRjLlA',
            ],
            [
                'accountId' => '57861',
                'listId' => '5fRi6V3LwkuEeHvDPRjLlA',
                'blacklistId' => 'mJPJiScYuUeCWA8Ct5MANg',
            ],
        ];
    }

    /**
     * @dataProvider provideAdapterLogin
     * @covers SarbacaneAdapter::__construct
     * @param string $accountId
     * @param string $listId
     * @param string $blacklistId
     */
    public function testConstructAdapter($accountId, $listId, $blacklistId = '')
    {
        $adapter = new SarbacaneAdapter($this->client, $accountId, $listId, $blacklistId);
        $this->assertEquals($accountId, $adapter->getAccountId());
        $this->assertEquals($listId, $adapter->getListId());
        $this->assertEquals($blacklistId, $adapter->getBlacklistId());
        if (empty($blacklistId)) {
            $this->assertFalse($adapter->isDeleteToBlacklist());
        } else {
            $this->assertTrue($adapter->isDeleteToBlacklist());
        }
    }

    /**
     * @dataProvider provideAdapterLogin
     * @covers SarbacaneAdapter::update
     * @param $accountId
     * @param $listId
     * @param string $blacklistId
     */
    public function testUpdate($accountId, $listId, $blacklistId = '')
    {
        $adapter = new SarbacaneAdapter($this->client, $accountId, $listId, $blacklistId);
        $this->assertTrue($adapter->update($this->email));
    }

    /**
     * @dataProvider provideAdapterLogin
     * @covers SarbacaneAdapter::insert
     * @param $accountId
     * @param $listId
     * @param string $blacklistId
     */
    public function testInsert($accountId, $listId, $blacklistId = '')
    {
        $adapter = new SarbacaneAdapter($this->client, $accountId, $listId, $blacklistId);
        $this->assertTrue($adapter->insert($this->email));
    }

    /**
     * @dataProvider provideAdapterLogin
     * @covers SarbacaneAdapter::delete
     * @param $accountId
     * @param $listId
     * @param string $blacklistId
     */
    public function testDelete($accountId, $listId, $blacklistId = '')
    {
        $adapter = new SarbacaneAdapter($this->client, $accountId, $listId, $blacklistId);
        $this->assertTrue($adapter->delete($this->email));
    }

    /**
     * @dataProvider provideAdapterLogin
     * @covers SarbacaneAdapter::blacklist
     * @param $accountId
     * @param $listId
     * @param string $blacklistId
     */
    public function testBlacklist($accountId, $listId, $blacklistId = '')
    {
        if (empty($blacklistId)) {
            $this->expectException(\InvalidArgumentException::class);
        }
        $adapter = new SarbacaneAdapter($this->client, $accountId, $listId, $blacklistId);
        $this->assertTrue($adapter->blacklist($this->email));
    }

    /**
     * @dataProvider provideAdapterLogin
     * @covers SarbacaneAdapter::unblacklist
     * @param $accountId
     * @param $listId
     * @param string $blacklistId
     */
    public function testUnblacklist($accountId, $listId, $blacklistId = '')
    {
        if (empty($blacklistId)) {
            $this->expectException(\InvalidArgumentException::class);
        }
        $adapter = new SarbacaneAdapter($this->client, $accountId, $listId, $blacklistId);
        $this->assertTrue($adapter->unblacklist($this->email));
    }
}
