<?php

namespace Cms\LinkResource\Provider;

use Cms\Cache\CacheHelperTrait;
use Cms\LinkResource\LinkResourceType;
use Move\Cache\CacheAwareInterface;
use POM\DomainObjectInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

/**
 * Class AbstractDelegateDataAdapter
 * @package Cms\LinkResource\Provider\CMS
 */
abstract class AbstractDelegateDataAdapter extends AbstractDelegateTypeAdapter implements
    DataAdapterAwareInterface,
    CacheAwareInterface,
    LoggerAwareInterface
{
    use LoggerAwareTrait;
    use CacheHelperTrait;

    /**
     * @param LinkResourceType|string $typeCode
     * @return \Cms\LinkResource\Provider\ProviderClientAdapterInterface
     * @throws \RuntimeException
     */
    abstract protected function getClientAdapter($typeCode) : ProviderClientAdapterInterface;

    /**
     * @param \POM\DomainObjectInterface $model
     * @return null|string
     */
    abstract protected function getLabelFromModel(DomainObjectInterface $model);

    /**
     * @return \Psr\Log\LoggerInterface
     */
    protected function getLogger() : LoggerInterface
    {
        return $this->logger ?: new NullLogger();
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param int $objectId
     * @return null|\POM\DomainObjectInterface
     */
    public function getData($typeCode, $objectId)
    {
        $result = null;
        try {
            // récuperation du client
            $client = $this->getClientAdapter($typeCode);
        } catch (\Exception $e) {
            return $result;
        }

        try {
            $classPart = explode('\\', __CLASS__);
            $cacheKey = vsprintf('%s^%s^%s', [
                array_pop($classPart),
                $typeCode,
                $objectId,
            ]);
            $result = $this->decorateWithCache(function () use ($client, $objectId) {
                $model = $client->fetchModel($objectId);
                return $model;
            }, $cacheKey, 60);
        } catch (\Exception $e) {
            $this->getLogger()->warning(
                'cannot retrieve data from api for ' . $typeCode . ' ' . $objectId,
                ['exception' => $e]
            );
        }
        return $result;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param int $objectId
     * @return string|null
     */
    public function getLabel($typeCode, $objectId)
    {
        $model = $this->getData($typeCode, $objectId);
        if ($model === null) {
            return null;
        }
        return $this->getLabelFromModel($model);
    }
}
