<?php

namespace Move\Template\Extensions;

use League\Plates\Engine;
use League\Plates\Extension\ExtensionInterface;
use Move\Advertiser\Google\DfpSmallBusiness\CollapsePolicy;
use Move\Advertiser\Google\DfpSmallBusiness\Size;
use Move\Advertiser\Google\DfpSmallBusiness\Slot;
use Move\Utils\Str;

/**
 * Class GoogleDfp
 * @package Move\Template\Extensions
 */
class GoogleDfp implements ExtensionInterface
{

    /** @var array */
    protected static $TARGETING = [];

    /** @var string */
    protected static $UNIT_PATH;

    /** @var bool */
    protected static $ENABLE = true;

    /** @var Slot[] */
    protected $slots = [];

    /** @var Slot[] */
    protected $usedSlots = [];

    /** @var string */
    protected $networkId;

    /** @var string */
    protected $collapsePolicy;

    /**
     * @return void
     */
    public static function enable()
    {
        static::$ENABLE = true;
    }

    /**
     * @return void
     */
    public static function disable()
    {
        static::$ENABLE = false;
    }

    /**
     * @param string $unitPath
     */
    public static function setUnitPath(string $unitPath)
    {
        static::$UNIT_PATH = $unitPath;
    }

    /**
     * @param string|array $name
     * @param string|array|null $value
     */
    public static function setTargeting($name, $value = null)
    {
        if (\is_string($name) && $value !== null) {
            static::$TARGETING[$name] = $value;
        } elseif (\is_array($name)) {
            foreach ($name as $key => $val) {
                static::setTargeting($key, $val);
            }
        }
    }

    /**
     * GoogleDfp constructor.
     * @param string $networkId
     * @param string $defaultUnitPath
     * @param array $targeting
     */
    public function __construct(string $networkId, string $defaultUnitPath, array $targeting = [])
    {
        static::setUnitPath($defaultUnitPath);
        static::setTargeting($targeting);

        $this->networkId = $networkId;
        $this->collapsePolicy = CollapsePolicy::EXPAND;
    }

    /**
     * @param \Move\Advertiser\Google\DfpSmallBusiness\CollapsePolicy|string $policy
     */
    public function setCollapsePolicy($policy)
    {
        $policy = (string)$policy;
        if (!CollapsePolicy::isValid($policy)) {
            throw new \InvalidArgumentException('collapse policy is invalid');
        }
        $this->collapsePolicy = $policy;
    }

    /**
     * @param string|bool|CollapsePolicy $policy
     * @return string
     */
    protected function convertCollapsePolicy($policy) : string
    {
        if (empty($policy) && is_scalar($policy)) {
            return '';
        }

        if (\is_bool($policy)) {
            return $policy ? 'true' : 'false';
        }

        $policy = (string)$policy;
        if (CollapsePolicy::isValid($policy)) {
            switch ($policy) {
                case CollapsePolicy::COLLAPSE:
                    return '';
                    break;
                case CollapsePolicy::EXPAND:
                    return 'true';
                    break;
                case CollapsePolicy::LEAVE:
                    return 'false';
                    break;
            }
        }

        return '';
    }

    /**
     * @param \League\Plates\Engine $engine
     */
    public function register(Engine $engine)
    {
        $engine->registerFunction('dfp_setup', [$this, 'setup']);
        $engine->registerFunction('dfp_display', [$this, 'display']);
        $engine->registerFunction('dfp_display_amp', [$this, 'displayAmp']);
    }

    /**
     * @param string $name
     * @param \Move\Advertiser\Google\DfpSmallBusiness\Slot $slot
     * @return $this
     */
    public function addSlot(string $name, Slot $slot) : self
    {
        if ($slot->sizeMapping) {
            $mappingName = $slot->sizeMapping->name ?: $name;
            $mappingName = Str::toAscii($mappingName, [], '_');
            $slot->sizeMapping->name = $mappingName;
        }
        $this->slots[$name] = $slot;
        return $this;
    }

    /**
     * @param string $name
     * @return \Move\Advertiser\Google\DfpSmallBusiness\Slot|null
     */
    protected function useSlot(string $name)
    {
        if (!isset($this->slots[$name])) {
            trigger_error('DFP_ERROR : no slot with name ' . $name, E_USER_WARNING);
            return null;
        }

        // récuperation du slot et sauvegarde
        $slot = $this->slots[$name];
        $this->usedSlots[$name] = $slot;
        unset($this->slots[$name]);

        return $slot;
    }

    /**
     * @return string
     */
    public function getFullUnitPath() : string
    {
        return '/' . $this->networkId . '/' . static::$UNIT_PATH;
    }

    /**
     * @param string $name
     * @param int $width
     * @param int $height
     * @return string
     */
    public function displayAmp(string $name, int $width, int $height) : string
    {
        if (!static::$ENABLE) {
            return '';
        }

        $slot = $this->useSlot($name);
        if ($slot === null) {
            return '<!-- DFP_ERROR : no slot with name ' . $name . ' -->';
        }

        // options suppl.
        $jsonOpts = [];
        $opts = '';

        // option de size
        if ($slot->sizes) {
            $opts .= 'data-multi-size="'
                . implode(',', array_map(function (Size $size) {
                    return $size->width . 'x' . $size->height;
                }, $slot->sizes))
                . '"' . "\n";
            $opts .= 'data-multi-size-validation="false"';
        }

        // option de targeting
        if ($slot->targeting) {
            $jsonOpts['targeting'] = $slot->targeting;
        }

        // json opts
        if ($jsonOpts) {
            $opts .= "json='" . json_encode($jsonOpts) . "'";
        }

        // chemin de la pub
        $path = $this->getFullUnitPath();

        $html = <<<EOF
<amp-ad width=$width height=$height
    type="doubleclick"
    $opts
    data-slot="$path">
</amp-ad>
EOF;
        return $html;
    }

    /**
     * @param string $name
     * @param string|null $divId
     * @param string|bool|CollapsePolicy|null $collapsePolicy
     * @return string
     */
    public function display(string $name, string $divId = null, $collapsePolicy = null) : string
    {
        if (!static::$ENABLE) {
            return '';
        }

        $slot = $this->useSlot($name);
        if ($slot === null) {
            return '<!-- DFP_ERROR : no slot with name ' . $name . ' -->';
        }

        // récuperation des donnée du slot
        $path = $this->getFullUnitPath();

        $divId = $divId ?: $slot->divId ?: $name;
        $mappingDef = '';

        if ($slot->sizes) {
            $sizes = json_encode($slot->sizes);
            $slotDef = "googletag.defineSlot('" . $path . "', " . $sizes . ", '" . $divId . "')";
        } else {
            $slotDef = "googletag.defineOutOfPageSlot('" . $path . "', '" . $divId . "')";
        }
        if ($slot->sizeMapping) {
            $mappingName = $slot->sizeMapping->name;
            $mappingDef .= 'var ' . $mappingName . ' = googletag.sizeMapping()';
            foreach ($slot->sizeMapping->adSizeCollection as [$navSize, $sizes]) {
                $mappingDef .= '.addSize(' . json_encode($navSize) . ', ' . json_encode($sizes) . ')';
            }
            $mappingDef .= '.build();';

            $slotDef .= '.defineSizeMapping(' . $mappingName . ')';
        }
        $slotDef .= '.addService(googletag.pubads())';
        foreach ($slot->targeting as $key => $val) {
            $valConvert = \is_array($val) ? json_encode($val) : "'" . $val . "'";
            $slotDef .= ".setTargeting('" . $key . "', " . $valConvert . ')';
        }

        // gestion de la methode de collapse
        if ($collapsePolicy === null) {
            $collapsePolicy = $this->convertCollapsePolicy($this->collapsePolicy);
        } else {
            $collapsePolicy = $this->convertCollapsePolicy($collapsePolicy);
        }
        if ($collapsePolicy === '') {
            $collapsePolicy = 'true,true';
        }
        $slotDef .= '.setCollapseEmptyDiv(' . $collapsePolicy . ')';
        $slotDef .= ';';

        $html = <<<EOF
<div id="$divId">
  <script>
    googletag.cmd.push(function() {
      $mappingDef
      $slotDef
      googletag.display('$divId'); 
    });
  </script>
</div>
EOF;
        return $html;
    }

    /**
     * @return string
     */
    public function setup() : string
    {
        if (!static::$ENABLE) {
            return '';
        }

        $targeting = '';
        foreach (static::$TARGETING as $key => $val) {
            $valConvert = \is_array($val) ? json_encode($val) : "'" . $val . "'";
            $targeting .= "googletag.pubads().setTargeting('" . $key . "', " . $valConvert . ");\n";
        }

        $collapsePolicy = $this->convertCollapsePolicy($this->collapsePolicy);

        $script = <<<EOF
<script async='async' src='https://www.googletagservices.com/tag/js/gpt.js'></script>
<script>
  var googletag = googletag || {};
  googletag.cmd = googletag.cmd || [];
</script>
<script>
    googletag.cmd.push(function(){
        // tags
        $targeting
      
        //identification du referer
        var adReferrer = document.referrer;
        (adReferrer.indexOf("google")>0) ? dfpReferrer = 'google' : dfpReferrer = 'other';
        googletag.pubads().setTargeting('referer',dfpReferrer);

        //gestion dynamique des pages de test
        var adTest = new URL(document.URL).searchParams.get("adtest");
        googletag.pubads().setTargeting('adtest',adTest);

        // opts & declanchement
        googletag.pubads().enableSingleRequest();
        googletag.pubads().setCentering(true);
        googletag.pubads().collapseEmptyDivs($collapsePolicy);
        googletag.enableServices();
    })
</script>
EOF;
        return $script;
    }
}
