<?php

namespace Move\Template\Html\Asset;

/**
 * Class Svg
 * @package Html\Asset
 */
class Svg implements AssetInterface
{

    /** @var array list of SVG files */
    protected $files = [];

    /** @var array list of directories to look for SVG files */
    protected $svgDirectories = [];

    /**
     * @param array $svgDirectories
     * @param array $aSvgFiles
     */
    public function __construct(array $svgDirectories, array $aSvgFiles = [])
    {
        $this->svgDirectories = $svgDirectories;
        foreach ($this->svgDirectories as $dir) {
            if (!is_dir($dir) || !is_readable($dir)) {
                throw new \InvalidArgumentException("directory $dir not exist or not readable");
            }
        }
        $this->addFiles($aSvgFiles);
    }

    /**
     * ajoute un svg
     * @param array $params [file => filename, keepStyles => bool]
     */
    public function includeSvg(array $params)
    {
        if (isset($params['file']) && $params['file'] !== null) {
            if (!isset($params['keepStyles'])
                || $params['keepStyles'] === null
                || filter_var($params['keepStyles'], FILTER_VALIDATE_BOOLEAN) === false
            ) {
                $params['keepStyles'] = false;
            }
            $this->addFile($params['file'], $params['keepStyles']);
        }
    }

    /**
     * @inheritdoc
     */
    public function render()
    {

        // pre check
        if (empty($this->files)) {
            return '';
        }

        $oFinalSvg = new \DOMDocument();
        $oFinalSvg->strictErrorChecking = false;

        // create root SVG element
        $oRootElement = $oFinalSvg->createElementNS('http://www.w3.org/2000/svg', 'svg');
        $oRootElement->setAttribute('style', 'display:none');

        // append root element to document
        $oFinalSvg->appendChild($oRootElement);

        foreach ($this->files as $aFile) {
            $sSvgFile = $aFile[0];
            $bKeepStyles = $aFile[1];

            $sSvgSource = file_get_contents($sSvgFile);
            $oSvgDom = new \DOMDocument();
            $oSvgDom->strictErrorChecking = false;
            $oSvgDom->loadXML($sSvgSource);

            // create a symbol element
            $oSymbol = $oFinalSvg->createElement('symbol');

            // defines the id
            $sSvgId = pathinfo($sSvgFile)['filename'];
            $oSymbol->setAttribute('id', 'svg-' . $sSvgId);

            // add additional attributes, without unwanted ones
            foreach ($oSvgDom->documentElement->attributes as $attribute) {
                /** @var \DOMAttr $attribute */
                if (!\in_array($attribute->nodeName, ['xmlns', 'width', 'height'])) {
                    $oSymbol->setAttribute($attribute->nodeName, $attribute->nodeValue);
                }
            }

            // append SVG child nodes to the symbol element
            foreach ($oSvgDom->documentElement->childNodes as $oChildNode) {
                /** @var \DOMElement $oChildNode */

                // import the child node into the document
                $oChildNode = $oFinalSvg->importNode($oChildNode, true);

                // remove unwanted child node attributes on path elements only
                if ($bKeepStyles === false && $oChildNode->nodeName === 'path') {
                    foreach ($oChildNode->attributes as $attribute) {
                        /** @var \DOMAttr $attribute */
                        // skip fill attributes having a `none` value
                        if ($attribute->nodeName === 'fill' && $attribute->nodeValue === 'none') {
                            continue;
                        }

                        // remove `fill` attributes where values start with `#`
                        if ($attribute->nodeName === 'fill' && preg_match('@^\#@', $attribute->nodeValue)) {
                            $oChildNode->removeAttributeNode($attribute);
                        }
                    }
                }
                $oSymbol->appendChild($oChildNode);
            }

            // append the symbol element
            $oRootElement->appendChild($oSymbol);
        }

        return $oFinalSvg->C14N(false, false);
    }

    /**
     * Ajoute un fichier SVG
     * @param $svgFile
     * @param bool $keepStyles
     * @return $this
     */
    public function addFile($svgFile, $keepStyles = false)
    {
        if (($svgFile = $this->fileExists($svgFile)) && $svgFile !== false) {
            $this->files[] = [$svgFile, $keepStyles];
        }
        return $this;
    }

    /**
     * Ajoute plusieurs fichiers SVG sous la forme [$svgFile, $keepStyles];
     * @param array $aSvgFiles
     * @return $this
     */
    public function addFiles(array $aSvgFiles)
    {
        foreach ($aSvgFiles as $file) {
            if (\is_array($file) && count($file) === 1) {
                $aFile = [$file[0], false];
            } elseif (!\is_array($file) && \is_string($file)) {
                $aFile = [$file, false];
            } else {
                $aFile = $file;
            }
            $this->addFile($aFile[0], $aFile[1]);
        }
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function getArrayCopy()
    {
        return [
            'directories' => $this->svgDirectories,
            'files' => $this->files,
        ];
    }

    /**
     * @param $sFile
     * @return bool|string
     */
    private function fileExists($sFile)
    {
        // if it's already a valid file path
        if (preg_match('@^/@', $sFile)) {
            return $sFile;
        }
        foreach ($this->svgDirectories as $sDirectory) {
            $sSvgFile = $sDirectory . $sFile . '.svg';
            if (is_file($sSvgFile)) {
                return $sSvgFile;
            }
        }
        return false;
    }

    /**
     * Effectue un rendu de l'objet en HTML
     * @return string
     */
    public function __toString()
    {
        return $this->render();
    }
}
