<?php


namespace Cms\Route;

/**
 * Class PathBuilder
 * @package Cms\Route
 */
class PathBuilder
{

    /** @var array */
    private $params;

    /** @var string */
    private $path;

    /**
     * RouteBuilder constructor.
     * @param string $path
     * @param array $params
     */
    public function __construct(string $path, array $params = [])
    {
        $this->params = $params;
        $this->path = $path;
    }

    /**
     * @param string $name
     * @param mixed $value
     * @param bool $overwrite
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function addParam(string $name, $value, bool $overwrite = true)
    {
        if (!$overwrite && isset($this->params[$name])) {
            throw new \InvalidArgumentException('param ' . $name . ' already exist');
        }
        $this->params[$name] = $value;
        return $this;
    }

    /**
     * @param array $params
     * @param bool $overwrite
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function mergeParams(array $params, bool $overwrite = true)
    {
        foreach ($params as $name => $value) {
            $this->addParam($name, $value, $overwrite);
        }
        return $this;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function removeParam(string $name)
    {
        unset($this->params[$name]);
        return $this;
    }

    /**
     * @return $this
     */
    public function clear()
    {
        $this->params = [];
        return $this;
    }

    /**
     * @return array
     */
    public function getArrayCopy() : array
    {
        return $this->params;
    }

    /**
     * @return string|null
     */
    public function build()
    {
        $userData = $this->params;
        // récuperartion de la route
        $path = $this->path;

        // nettoyage de la chaine
        if (preg_match_all('@{([a-z0-9_-]+)[^}]+}@', $path, $matches) && !empty($matches[1])) {
            $empty = '{#}';
            foreach ($matches[1] as $key) {
                $value = $userData[$key] ?? $empty;
                $path = preg_replace('@{' . $key . '[^}]+}@', $value, $path);
            }

            // on supprime de la chaine toute les chaines non remplacé
            $path = preg_replace('@\[[^{]*' . $empty . '[^\]]*\]@', '', $path);

            // si il reste des chaine non remplacé c pas bon
            if (strpos($path, $empty) !== false) {
                trigger_error(
                    'some part of path are not replaced in ' . $this->path,
                    E_USER_WARNING
                );
                return null;
            }

            // on del les [ ] si remplacé
            $path = preg_replace('@\[|\]@', '', $path);
        }
        return $path;
    }

    /**
     * @return null|string
     */
    public function __invoke()
    {
        return $this->build();
    }

    /**
     * @return string
     */
    public function __toString() : string
    {
        return $this->build() ?: '';
    }
}
