<?php


namespace Cms\Search\Provider\Elastica;

use Cms\Search\ResultSetInterface;
use Elastica\Result;
use Elastica\ResultSet;
use Move\Iterator\MapIterator;

/**
 * Class ElasticSearchResultSet
 * @package Cms\Search
 */
class ElasticSearchResultSet extends MapIterator implements ResultSetInterface
{

    /**
     * @var ResultSet
     */
    private $resultSet;

    /**
     * ElasticSearchResultSet constructor.
     * @param ResultSet $resultSet
     */
    public function __construct(ResultSet $resultSet)
    {
        parent::__construct($resultSet, function (Result $result) {
            return new ElasticSearchResultElement($result);
        });
        $this->resultSet = $resultSet;
    }

    /**
     * @inheritdoc
     */
    public function count() : int
    {
        return $this->resultSet->count();
    }

    /**
     * @return int
     */
    public function getTotalHits() : int
    {
        return $this->resultSet->getTotalHits();
    }

    /**
     * @return ResultSet
     */
    public function getElasticResultSet() : ResultSet
    {
        return $this->resultSet;
    }

    /**
     * @param string $name
     * @return \Cms\Search\Provider\Elastica\ElasticSearchAggregation
     */
    public function getElasticAggregation(string $name) : ElasticSearchAggregation
    {
        $aggs = $this->resultSet->getAggregations();
        $aggData = $this->searchInAggs($aggs, $name);
        return new ElasticSearchAggregation($aggData);
    }

    /**
     * @param string $name
     * @param string $metricName
     * @return mixed|null
     */
    public function getElasticAggregationMetric(string $name, string $metricName = 'value')
    {
        $aggs = $this->resultSet->getAggregations();
        $aggData = $this->searchInAggs($aggs, $name);
        return $aggData[$metricName] ?? null;
    }

    /**
     * @param array $aggs
     * @param string $searchName
     * @return array
     */
    private function searchInAggs(array $aggs, string $searchName) : array
    {
        // recherche rapide
        if (isset($aggs[$searchName])) {
            return $aggs[$searchName];
        }
        // recherche approfondi
        foreach ($aggs as $aggName => $aggData) {
            if (isset($aggData['doc_count']) && $aggData['doc_count'] === 0) {
                continue;
            }
            if (!\is_array($aggData)) {
                continue;
            }
            foreach ($aggData as $tmpName => $aggChild) {
                if (\is_array($aggChild)) {
                    if ((string)$tmpName === $searchName) {
                        return $aggChild;
                    }
                    $aggFind = $this->searchInAggs($aggChild, $searchName);
                    if (!empty($aggFind)) {
                        return $aggFind;
                    }
                }
            }
        }
        return [];
    }
}
