<?php

namespace Move\Jato\Mapper;

use GuzzleHttp\Client;
use Move\Command\Exception\BadParameterException;
use Move\Http\Client\GuzzleClientAdapter;
use Move\Jato\JatoClient;
use Move\Jato\Model\BodyModel;
use PHPUnit\Framework\TestCase;

/**
 * Class BodyMapperTest
 * @package Move\Jato\Mapper
 */
class BodyMapperTest extends TestCase
{
    /**
     * @return BodyMapper
     */
    public function testConstruct() : BodyMapper
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );

        $mapper = new BodyMapper($client);
        $clientTest = $mapper->getClient();

        $this->assertInstanceOf(JatoClient::class, $clientTest, 'Client Error');
        $this->assertEquals($client, $clientTest, 'Client not equal');

        return $mapper;
    }

    /**
     * @depends testConstruct
     * @param BodyMapper $mapper
     */
    public function testGetClient(BodyMapper $mapper)
    {
        $result = $mapper->getClient();
        $this->assertInstanceOf(JatoClient::class, $result, 'Not a JatoClient');
    }

    /**
     * @return array
     */
    public function handleDataProvider() : array
    {
        $data = <<<'JSONDOC'
[
  {
      "databaseName": "SSCF_CS2002",
    "name": "France",
    "flagUrl": "https://sslphotos.jato.com/Flags/F.png",
    "bodies": [
      {
          "makeNameGlobal": "ASTON MARTIN",
        "makeNameLocal": "Aston Martin",
        "makeKey": "Aston Martin",
        "makeNameToDisplay": "Aston Martin",
        "modelNameGlobal": "DB11",
        "modelNameLocal": "DB11",
        "modelKey": "DB11",
        "modelNameToDisplay": "DB11",
        "modelYear": "2017",
        "modelYearToDisplay": "2017",
        "numberOfDoors": "2",
        "numberOfDoorsToDisplay": "2",
        "shortBodyName": "coupe",
        "bodyPhotoUrl": "https://sslphotos.jato.com/PHOTO100/SSCF/ASTON MARTIN/DB11/2017/2CO.JPG",
        "bodyCode": "CO",
        "bodyCodeToDisplay": "CO",
        "minPrice": 180673,
        "maxPrice": 180673,
        "minPriceToDisplay": "£180,674",
        "maxPriceToDisplay": "£180,674",
        "drivenWheels": "R",
        "drivenWheelsToDisplay": "rear"
      }
    ]
  }
]
JSONDOC;
        $return = json_decode($data, true);
        return [$return];
    }

    /**
     * @dataProvider handleDataProvider
     * @param array $data
     */
    public function testHandleData(array $data)
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );
        $mapper = new BodyMapper($client);

        $data = $data['bodies'];
        $return = [];
        foreach ($data as $body) {
            $bodyModel = $mapper->handleData($body);
            $return[] = $bodyModel;
        }
        $this->assertNotEmpty($return, 'Array empty');

        foreach ($return as $bodyTest) {
            $this->assertInstanceOf(BodyModel::class, $bodyTest, 'Not BodyModel');
        }
    }

    /**
     * @depends testConstruct
     * @param BodyMapper $mapper
     */
    public function testFetchAllBodiesForModel(BodyMapper $mapper)
    {
        $database = 'SSCF_CS2002';
        $makeKey = 'Aston Martin';
        $modelKey = 'DB11';
        $modelYear = '2017';

        $result = $mapper->fetchAllBodiesForModel($database, $makeKey, $modelKey, $modelYear);
        $this->assertNotEmpty($result, 'FetchAllBodies failed');
        foreach ($result as $body) {
            $this->assertInstanceOf(BodyModel::class, $body, 'Not BodyModel');
        }
    }

    /**
     * @depends testConstruct
     * @param BodyMapper $mapper
     */
    public function testFetchAllBodiesForModelInvalid(BodyMapper $mapper)
    {
        $this->expectException(BadParameterException::class);

        $database = null;
        $makeKey = null;
        $modelKey = null;
        $modelYear = null;
        $mapper->fetchAllBodiesForModel($database, $makeKey, $modelKey, $modelYear);
    }

}