<?php


namespace Cms\Client\Geodata\Helper;

use Cms\Cache\CacheFactory;
use Cms\Client\CmsApiIndexIterator;
use Cms\Client\Command\ExceptionDecoratorInterface;
use Cms\Client\Command\Geodata\Command\FetchAdminCommand;
use Cms\Client\Command\Geodata\Command\FetchAllAdminByFilterCommand;
use Cms\Client\Command\Geodata\Command\FetchCityCommand;
use Cms\Client\Command\Geodata\Command\FetchCountryCommand;
use Cms\Client\Geodata\Model\AdminRestfulModel;
use Cms\Client\Geodata\Model\CityRestfulModel;
use Cms\Client\Service\GeodataService;
use Cms\Model\Geodata\CountryModel;
use Move\Command\Exception\CommandException;
use Move\Command\Exception\NotFoundException;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Cache\InvalidArgumentException;

/**
 * Trait GeodataHelperTrait
 * @package Cms\Client\Geodata\Helper
 */
trait GeodataHelperTrait
{

    /**
     * @return \Cms\Client\Service\GeodataService
     * @throws \DomainException
     */
    abstract public function getGeodataService() : GeodataService;

    /**
     * @var \Psr\Cache\CacheItemPoolInterface
     */
    protected $geoadminCachePool;


    /**
     * @param \Cms\Client\Geodata\Model\AdminRestfulModel $parentModel
     * @param string|null $countryCode
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Geodata\Model\AdminRestfulModel[]
     * @throws \Throwable
     */
    protected function getGeodataAdminChilds(
        AdminRestfulModel $parentModel,
        string $countryCode = null,
        ExceptionDecoratorInterface $decorator = null
    ) : array {
        $depts = $this->getGeodataAdminLevelList(2, $countryCode ?: $parentModel->country_alpha2, true, $decorator);
        // filtrage sur les depts de la region courante
        $depts = array_filter(
            $depts,
            function (AdminRestfulModel $dept) use ($parentModel) {
                return (int)$dept->parent_id === (int)$parentModel->id;
            }
        );
        return array_values($depts);
    }

    /**
     * @param int $level
     * @param string|null $countryCode
     * @param bool $withDomtom
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Geodata\Model\AdminRestfulModel[]|null
     * @throws \Throwable
     */
    protected function getGeodataAdminLevelList(
        int $level,
        string $countryCode = null,
        bool $withDomtom = false,
        ExceptionDecoratorInterface $decorator = null
    ) {
        // creation du cche pool
        if (!$this->geoadminCachePool instanceof CacheItemPoolInterface) {
            $this->geoadminCachePool = CacheFactory::createLocalPool();
        }

        $countryCode = $countryCode ?: 'FR';
        try {
            $cacheItem = $this->geoadminCachePool->getItem(
                'GeoadminCache^listadminlevel^' . $level . '^' . $countryCode
            );
        } catch (InvalidArgumentException $e) {
            $cacheItem = null;
        }
        if ($cacheItem && $cacheItem->isHit()) {
            $depts = $cacheItem->get();
        } else {
            try {
                $depts = $this->getGeodataService()->getAdminClient()->fetchAllByCountry(
                    $countryCode,
                    null,
                    $level,
                    -1
                );
                $depts = iterator_to_array($depts ?: new \ArrayIterator(), false);
            } catch (\Throwable $e) {
                if ($decorator !== null) {
                    \call_user_func($decorator->getThrowableDecorator($e));
                }
                throw $e;
            }

            // mise en cache
            if ($cacheItem) {
                $cacheItem
                    ->expiresAfter(new \DateInterval('P1D'))
                    ->set($depts);
                $this->geoadminCachePool->save($cacheItem);
            }
        }

        // filtre les domtom
        if (!$withDomtom && $countryCode === 'FR') {
            $depts = array_filter($depts, function ($admin) {
                return \in_array($admin['code_simple'], ['2A', '2B'], true)
                    || (is_numeric($admin['code_simple']) && $admin['code_simple'] < 99);
            });
        }

        return array_values($depts);
    }

    /**
     * @param string|int|null $identifier
     * @param string|null $admin1Code
     * @param string|null $admin2Code
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Geodata\Model\CityRestfulModel
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getGeodataCityModel(
        $identifier,
        string $admin1Code = null,
        string $admin2Code = null,
        ExceptionDecoratorInterface $decorator = null
    ) : CityRestfulModel {
        $commandBus = $this->getGeodataService()->getCommandBus();
        if (\is_string($identifier) && ($admin1Code || $admin2Code)) {
            $command = new FetchCityCommand(null);
            $command->setSlugPart($identifier);
            if ($admin1Code) {
                $command->setAdmin1Code($admin1Code);
            }
            if ($admin2Code) {
                $command->setAdmin2Code($admin2Code);
            }
        } else {
            $command = new FetchCityCommand($identifier);
        }
        try {
            $cityModel = $commandBus->handle($command);
        } catch (NotFoundException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getNotFoundDecorator($e));
            }
            throw $e;
        } catch (CommandException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getThrowableDecorator($e));
            }
            throw $e;
        }
        return $cityModel;
    }

    /**
     * @param string|int $identifier Country alpha code
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Model\Geodata\CountryModel
     * @throws \DomainException
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getGeodataCountryModel(
        $identifier,
        ExceptionDecoratorInterface $decorator = null
    ) : CountryModel {
        try {
            $commandBus = $this->getGeodataService()->getCommandBus();
            $command = new FetchCountryCommand($identifier);
            $countryModel = $commandBus->handle($command);
        } catch (NotFoundException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getNotFoundDecorator($e));
            }
            throw $e;
        } catch (CommandException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getThrowableDecorator($e));
            }
            throw $e;
        }
        return $countryModel;
    }

    /**
     * @param string|int $identifier Admin concat code or id
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Geodata\Model\AdminRestfulModel
     * @throws \DomainException
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getGeodataAdminModel(
        $identifier,
        ExceptionDecoratorInterface $decorator = null
    ) : AdminRestfulModel {
        try {
            $commandBus = $this->getGeodataService()->getCommandBus();
            $command = new FetchAdminCommand($identifier);
            $adminModel = $commandBus->handle($command);
        } catch (NotFoundException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getNotFoundDecorator($e));
            }
            throw $e;
        } catch (CommandException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getThrowableDecorator($e));
            }
            throw $e;
        }
        return $adminModel;
    }

    /**
     * @param string $stateSlug
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Geodata\Model\AdminRestfulModel|null
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getStateModelBySlug(
        string $stateSlug,
        ExceptionDecoratorInterface $decorator = null
    ) {
        try {
            $cmd = new FetchAllAdminByFilterCommand('FR', null, 1);
            $cmd->setAdminSlug($stateSlug);
            $admins = $this->getGeodataService()->getCommandBus()->handle($cmd);
        } catch (NotFoundException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getNotFoundDecorator($e));
            }
            throw $e;
        } catch (CommandException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getThrowableDecorator($e));
            }
            throw $e;
        }
        if (!$admins instanceof CmsApiIndexIterator
            || !$admins->getFirst() instanceof AdminRestfulModel
        ) {
            return null;
        }
        return $admins->getFirst();
    }

    /**
     * @param string $countySlug
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Geodata\Model\AdminRestfulModel|null
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getCountyModelBySlug(
        string $countySlug,
        ExceptionDecoratorInterface $decorator = null
    ) {
        try {
            $cmd = new FetchAllAdminByFilterCommand('FR', null, 2);
            $cmd->setAdminSlug($countySlug);
            $admins = $this->getGeodataService()->getCommandBus()->handle($cmd);
        } catch (NotFoundException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getNotFoundDecorator($e));
            }
            throw $e;
        } catch (CommandException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getThrowableDecorator($e));
            }
            throw $e;
        }
        if (!$admins instanceof CmsApiIndexIterator
            || !$admins->getFirst() instanceof AdminRestfulModel
        ) {
            return null;
        }
        return $admins->getFirst();
    }
}
