<?php


namespace Move\Search;

use Elastica\Bulk\ResponseSet;
use Elastica\Document;
use Elastica\Exception\NotFoundException;
use Elastica\Response;
use Elastica\Type\Mapping;

/**
 * Class ElasticDocumentHandler
 * @package Move\Search
 */
class ElasticDocumentHandler
{

    /** @var \Elastica\Type */
    private $type;

    /**
     * ElasticDocumentHandler constructor.
     * @param \Elastica\Type\Mapping $mapping
     */
    public function __construct(Mapping $mapping)
    {
        $this->type = $mapping->getType();
    }

    /**
     * @param Document[] $documents
     * @return \Elastica\Bulk\ResponseSet
     */
    public function indexDocuments($documents) : ResponseSet
    {
        return $this->type->addDocuments($documents);
    }

    /**
     * @param Document $document
     * @param bool $checkForUpdate
     * @return \Elastica\Response
     * @throws \Elastica\Exception\InvalidException
     */
    public function indexDocument(Document $document, $checkForUpdate = true) : Response
    {
        if ($checkForUpdate === true) {
            $getDocument = $this->getDocumentById($document->getId());
            if (!$getDocument) {
                return $this->type->addDocument($document);
            }
            return $this->type->updateDocument($document);
        }

        return $this->type->addDocument($document);
    }

    /**
     * @param int $articleId
     * @return Document|null
     */
    public function getDocumentById($articleId)
    {
        try {
            $document = $this->type->getDocument($articleId);
            return $document;
        } catch (NotFoundException $e) {
            return null;
        } catch (\Exception $e) {
            trigger_error(
                'an error occured while getting the document ' . $articleId . ' : ' . $e->getMessage(),
                E_USER_WARNING
            );
        }
        return null;
    }

    /**
     * @return \Elastica\Response
     */
    public function refreshIndex() : Response
    {
        return $this->type->getIndex()->refresh();
    }

    /**
     * @param int $articleId
     * @return \Elastica\Bulk\ResponseSet|null
     */
    public function deleteDocumentById($articleId)
    {
        $document = $this->getDocumentById($articleId);
        if ($document) {
            return $this->type->deleteDocuments([$document]);
        }
        return null;
    }
}
