<?php

namespace Move\Jato\Mapper;

use GuzzleHttp\Client;
use Move\Command\Exception\BadParameterException;
use Move\Http\Client\GuzzleClientAdapter;
use Move\Jato\JatoClient;
use Move\Jato\Model\ModelModel;
use PHPUnit\Framework\TestCase;

/**
 * Class ModelMapperTest
 * @package Move\Jato\Mapper
 */
class ModelMapperTest extends TestCase
{
    /**
     * @return ModelMapper
     */
    public function testConstruct() : ModelMapper
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );

        $mapper = new ModelMapper($client);
        $clientTest = $mapper->getClient();

        $this->assertInstanceOf(JatoClient::class, $clientTest, 'Client error');
        $this->assertEquals($client, $clientTest, 'Client not equal');

        return $mapper;
    }

    /**
     * @depends testConstruct
     * @param ModelMapper $mapper
     */
    public function testGetClient(ModelMapper $mapper)
    {
        $client = $mapper->getClient();
        $this->assertInstanceOf(JatoClient::class, $client, 'Error getClient');
    }

    /**
     * @return array
     */
    public function handleDataProvider() : array
    {
        $data = '[
    {
        "databaseName": "SSCF_CS2002",
        "name": "France",
        "flagUrl": "https://sslphotos.jato.com/Flags/F.png",
        "models": [
            {
                "makeNameGlobal": "ASTON MARTIN",
                "makeNameLocal": "Aston Martin",
                "makeKey": "Aston Martin",
                "makeNameToDisplay": "Aston Martin",
                "modelNameGlobal": "DB11",
                "modelNameLocal": "DB11",
                "modelKey": "DB11",
                "modelNameToDisplay": "DB11",
                "modelYear": "2017",
                "modelYearToDisplay": "2017",
                "modelPhotoUrl": "https://sslphotos.jato.com/PHOTO100/SSCF/ASTON MARTIN/DB11/2017/2CO.JPG",
                "minPrice": 180673,
                "maxPrice": 180673,
                "minPriceToDisplay": "£180,674",
                "maxPriceToDisplay": "£180,674"
            },
            {
                "makeNameGlobal": "ASTON MARTIN",
                "makeNameLocal": "Aston Martin",
                "makeKey": "Aston Martin",
                "makeNameToDisplay": "Aston Martin",
                "modelNameGlobal": "RAPIDE",
                "modelNameLocal": "RAPIDE",
                "modelKey": "RAPIDE",
                "modelNameToDisplay": "RAPIDE",
                "modelYear": "2015",
                "modelYearToDisplay": "2015",
                "modelPhotoUrl": "https://sslphotos.jato.com/PHOTO100/SSCF/ASTON MARTIN/RAPIDE/2015/5HA.JPG",
                "minPrice": 168784,
                "maxPrice": 168784,
                "minPriceToDisplay": "£168,785",
                "maxPriceToDisplay": "£168,785"
            }
        ]
    }
    ]';
        $return = json_decode($data, true);
        return [$return];
    }

    /**
     * @dataProvider handleDataProvider
     * @param array $data
     */
    public function testHandleData(array $data)
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );
        $mapper = new ModelMapper($client);

        $data = $data['models'];
        $return = [];
        foreach ($data as $model) {
            $modelModel = $mapper->handleData($model);
            array_push($return, $modelModel);
        }
        $this->assertNotEmpty($return, 'Array empty');

        foreach ($return as $modelTest) {
            $this->assertInstanceOf(ModelModel::class, $modelTest, 'Not ModelModel');
        }
    }

    /**
     * @depends testConstruct
     * @param ModelMapper $mapper
     */
    public function testFetchAllForMake(ModelMapper $mapper)
    {
        $dataBase = 'SSCF_CS2002';
        $makeKey = 'Aston Martin';

        $result = $mapper->fetchAllForMake($dataBase, $makeKey);
        $this->assertInternalType('array', $result, 'Not an array');
        $this->assertNotEmpty($result, 'FetchAllForMake failed');
        foreach ($result as $model) {
            $this->assertInstanceOf(ModelModel::class, $model, 'Not a ModelModel');
        }
    }

    /**
     * @depends testConstruct
     * @param ModelMapper $mapper
     */
    public function testFetchAllModels(ModelMapper $mapper)
    {
        $dataBase = 'SSCF_CS2002';
        $makeKey = 'Aston Martin';
        $modelKey = 'DB11';

        $result = $mapper->fetchAllModels($dataBase, $makeKey, $modelKey);
        $this->assertInternalType('array', $result, 'Not an array');
        $this->assertNotEmpty($result, 'FetchAllModels failed');

        foreach ($result as $model) {
            $this->assertInstanceOf(ModelModel::class, $model, 'Not a ModelModel');
        }
    }

    /**
     * @depends testConstruct
     * @param ModelMapper $mapper
     */
    public function testFetchAllModelsInvalid(ModelMapper $mapper)
    {
        $this->expectException(BadParameterException::class);

        $dataBase = 'SSCF_CS2002';
        $makeKey = 'Aston Martin';
        $modelKey = null;
        $mapper->fetchAllModels($dataBase, $makeKey, $modelKey);
    }

    /**
     * @depends testConstruct
     * @param ModelMapper $mapper
     */
    public function testGetModel(ModelMapper $mapper)
    {
        $dataBase = 'SSCF_CS2002';
        $makeKey = 'Aston Martin';
        $modelKey = 'DB11';

        $expected = $mapper->fetchAllModels($dataBase, $makeKey, $modelKey);
        $result = $mapper->getModel($dataBase, $makeKey, $modelKey);

        $this->assertEquals($expected[0], $result, 'Not Equal');
    }

}